% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clade_continuous.R
\name{clade_continuous}
\alias{clade_continuous}
\title{Influential Clade Detection - Trait Evolution Continuous Characters}
\usage{
clade_continuous(data, phy, model, trait.col, clade.col, n.species = 5,
  n.sim = 20, bounds = list(), n.cores = NULL, track = TRUE, ...)
}
\arguments{
\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{model}{The evolutionary model (see Details).}

\item{trait.col}{The column in the provided data frame which specifies the
trait to analyse (which should be a factor with two level)}

\item{clade.col}{The column in the provided data frame which specifies the
clades (a character vector with clade names).}

\item{n.species}{Minimum number of species in a clade for the clade to be
included in the leave-one-out deletion analysis. Default is \code{5}.}

\item{n.sim}{Number of simulations for the randomization test.}

\item{bounds}{settings to constrain parameter estimates. See \code{\link[geiger]{fitContinuous}}}

\item{n.cores}{number of cores to use. If 'NULL', number of cores is detected.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{\link[geiger]{fitContinuous}}}
}
\value{
The function \code{tree_continuous} returns a list with the following
components:

\code{call}: The function call

\code{data}: The original full data frame.

\code{full.model.estimates}: Parameter estimates (rate of evolution \code{sigsq}
and where applicable \code{optpar}), root state \code{z0}, 
AICc for the full model without deleted clades.

\code{sensi.estimates}: Parameter estimates (sigsq and optpar), (percentual) difference 
in parameter estimate compared to the full model (DIFsigsq, sigsq.perc, DIFoptpar, optpar.perc),
AICc and z0 for each repeat with a clade removed.

\code{null.dist}: A data frame with estimates for the null distributions
for all clades analysed.

\code{errors}: Clades where deletion resulted in errors.

\code{clade.col}: Which column was used to specify the clades?

\code{optpar}: Transformation parameter used (e.g. \code{lambda}, \code{kappa} etc.)
}
\description{
Fits models for trait evolution of continuous characters, 
detecting influential clades
}
\details{
This function sequentially removes one clade at a time,
fits a model of continuous character evolution using \code{\link[geiger]{fitContinuous}}, 
repeats this many times (controlled by \code{n.sim}), stores the results and calculates 
the effects on model parameters Currently, only binary continuous traits are supported. 

Additionally, to account for the influence of the number of species on each 
clade (clade sample size), this function also estimates a null distribution
expected for the number of species in a given clade. This is done by fitting
 models without the same number of species as in the given clade.The number of 
 simulations to be performed is set by 'n.sim'. To test if the 
 clade influence differs from the null expectation for a clade of that size, 
 a randomization test can be performed using 'summary(x)'. 

Different evolutionary models from \code{fitContinuous} can be used, i.e. \code{BM},\code{OU},
\code{EB}, \code{trend}, \code{lambda}, \code{kappa}, \code{delta} and \code{drift}.

See \code{\link[geiger]{fitContinuous}} for more details on evolutionary models. 

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontshow{
#Load data:
data("primates")
#Model trait evolution accounting for influential clades
clade_cont<-clade_continuous(data=primates$data,phy = primates$phy[[1]],model="BM",
trait.col = "adultMass",clade.col="family",n.sim=1,n.species=20,n.cores = 2,track=TRUE)
}
\dontrun{
data("primates")
#Model trait evolution accounting for phylogenetic uncertainty
clade_cont<-clade_continuous(data=primates$data,phy = primates$phy[[1]], model="OU",
trait.col = "adultMass",clade.col="family",n.sim=30,n.species=10,n.cores = 2,track=TRUE)
#Print summary statistics
summary(clade_cont)
sensi_plot(clade_cont,graph="all")
sensi_plot(clade_cont,clade="Cercopithecidae",graph = "sigsq")
sensi_plot(clade_cont,clade="Cercopithecidae",graph = "optpar")
#Change the evolutionary model, tree transformation or minimum number of species per clade
clade_cont2<-clade_continuous(data=primates$data,phy = primates$phy[[1]],model="delta",
trait.col = "adultMass",clade.col="family",n.sim=30,n.species=5,n.cores = 2,track=TRUE)
summary(clade_cont2)
sensi_plot(clade_cont2)
clade_cont3<-clade_continuous(data=primates$data,phy = primates$phy[[1]],model="BM",
trait.col = "adultMass",clade.col="family",n.sim=30,n.species=5,n.cores = 2,track=TRUE)
summary(clade_cont3)
sensi_plot(clade_cont3,graph="sigsq")
}
}
\references{
Yang Z. 2006. Computational Molecular Evolution. Oxford University Press: Oxford. 

Harmon Luke J, Jason T Weir, Chad D Brock, Richard E Glor, and Wendell Challenger. 2008.
GEIGER: investigating evolutionary radiations. Bioinformatics 24:129-131.
}
\seealso{
\code{\link[geiger]{fitContinuous}}
}
\author{
Gijsbert Werner & Gustavo Paterno
}
