% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_mmm.R
\name{build_mmm}
\alias{build_mmm}
\title{Build a Mixture Markov Model}
\usage{
build_mmm(
  observations,
  n_clusters,
  transition_probs,
  initial_probs,
  formula = NULL,
  data = NULL,
  coefficients = NULL,
  cluster_names = NULL,
  ...
)
}
\arguments{
\item{observations}{An \code{stslist} object (see \code{\link[TraMineR:seqdef]{TraMineR::seqdef()}}) containing
the sequences.}

\item{n_clusters}{A scalar giving the number of clusters/submodels
(not used if starting values for model parameters are given with
\code{initial_probs} and \code{transition_probs}).}

\item{transition_probs}{A list of matrices of transition
probabilities for submodels of each cluster.}

\item{initial_probs}{A list which contains vectors of initial state
probabilities for submodels of each cluster.}

\item{formula}{Optional formula of class \code{\link[=formula]{formula()}} for the
mixture probabilities. Left side omitted.}

\item{data}{A data frame containing the variables used in the formula.
Ignored if no formula is provided.}

\item{coefficients}{An optional \eqn{k x l} matrix of regression coefficients for
time-constant covariates for mixture probabilities, where \eqn{l} is the number
of clusters and \eqn{k} is the number of covariates. A logit-link is used for
mixture probabilities. The first column is set to zero.}

\item{cluster_names}{A vector of optional names for the clusters.}

\item{...}{Additional arguments to \code{simulate_transition_probs}.}
}
\value{
Object of class \code{mhmm} with following elements:
\itemize{
\item \code{observations}\cr State sequence object or a list of such containing the data.
\item \code{transition_probs}\cr A matrix of transition probabilities.
\item \code{emission_probs}\cr A matrix or a list of matrices of emission probabilities.
\item \code{initial_probs}\cr A vector of initial probabilities.
\item \code{coefficients}\cr A matrix of parameter coefficients for covariates (covariates in rows, clusters in columns).
\item \code{X}\cr Covariate values for each subject.
\item \code{cluster_names}\cr Names for clusters.
\item \code{state_names}\cr Names for hidden states.
\item \code{symbol_names}\cr Names for observed states.
\item \code{channel_names}\cr Names for channels of sequence data
\item \code{length_of_sequences}\cr (Maximum) length of sequences.
\item \code{sequence_lengths}\cr A vector of sequence lengths.
\item \code{n_sequences}\cr Number of sequences.
\item \code{n_symbols}\cr Number of observed states (in each channel).
\item \code{n_states}\cr Number of hidden states.
\item \code{n_channels}\cr Number of channels.
\item \code{n_covariates}\cr Number of covariates.
\item \code{n_clusters}\cr Number of clusters.
}
}
\description{
Function \code{\link[=build_mmm]{build_mmm()}} is a shortcut for constructing a mixture Markov
model as a restricted case of an \code{mhmm} object.
}
\examples{


# Define sequence data
data("mvad", package = "TraMineR")
mvad_alphabet <- c(
  "employment", "FE", "HE", "joblessness", "school",
  "training"
)
mvad_labels <- c(
  "employment", "further education", "higher education",
  "joblessness", "school", "training"
)
mvad_scodes <- c("EM", "FE", "HE", "JL", "SC", "TR")
mvad_seq <- seqdef(mvad, 15:86,
  alphabet = mvad_alphabet, states = mvad_scodes,
  labels = mvad_labels, xtstep = 6
)

# Initialize the MMM
set.seed(123)
mmm_mvad <- build_mmm(
  observations = mvad_seq,
  n_clusters = 2,
  formula = ~male, data = mvad
)

\dontrun{
# Estimate model parameters
mmm_mvad <- fit_model(mmm_mvad)$model

# Plot model (both clusters in the same plot)
require(igraph)
plot(mmm_mvad,
  interactive = FALSE,
  # Modify legend position and properties
  with.legend = "right", legend.prop = 0.3, cex.legend = 1.2,
  # Define vertex layout
  layout = layout_as_star,
  # Modify edge properties
  edge.label = NA, edge.arrow.size = 0.8, edge.curved = 0.2,
  # Modify vertex label positions (initial probabilities)
  vertex.label.pos = c("left", "right", "right", "left", "left", "right")
)

# Summary of the MMM
summary(mmm_mvad)
}
}
\seealso{
\code{\link[=fit_model]{fit_model()}} for estimating model parameters;
\code{\link[=summary.mhmm]{summary.mhmm()}} for a summary of a mixture model;
\code{\link[=separate_mhmm]{separate_mhmm()}} for organizing an \code{mhmm} object into a list of
separate \code{hmm} objects; and \code{\link[=plot.mhmm]{plot.mhmm()}} for plotting
mixture models.
}
