% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shapviz.R
\name{shapviz}
\alias{shapviz}
\alias{shapviz.default}
\alias{shapviz.matrix}
\alias{shapviz.xgb.Booster}
\alias{shapviz.lgb.Booster}
\alias{shapviz.explain}
\alias{shapviz.treeshap}
\alias{shapviz.predict_parts}
\alias{shapviz.shapr}
\alias{shapviz.kernelshap}
\alias{shapviz.H2ORegressionModel}
\alias{shapviz.H2OBinomialModel}
\alias{shapviz.H2OModel}
\title{Initialize "shapviz" Object}
\usage{
shapviz(object, ...)

\method{shapviz}{default}(object, ...)

\method{shapviz}{matrix}(object, X, baseline = 0, collapse = NULL, S_inter = NULL, ...)

\method{shapviz}{xgb.Booster}(
  object,
  X_pred,
  X = X_pred,
  which_class = NULL,
  collapse = NULL,
  interactions = FALSE,
  ...
)

\method{shapviz}{lgb.Booster}(object, X_pred, X = X_pred, which_class = NULL, collapse = NULL, ...)

\method{shapviz}{explain}(object, X = NULL, baseline = NULL, collapse = NULL, ...)

\method{shapviz}{treeshap}(
  object,
  X = object[["observations"]],
  baseline = 0,
  collapse = NULL,
  ...
)

\method{shapviz}{predict_parts}(object, ...)

\method{shapviz}{shapr}(object, X = object[["x_test"]], collapse = NULL, ...)

\method{shapviz}{kernelshap}(object, X = object[["X"]], which_class = NULL, collapse = NULL, ...)

\method{shapviz}{H2ORegressionModel}(object, X_pred, X = as.data.frame(X_pred), collapse = NULL, ...)

\method{shapviz}{H2OBinomialModel}(object, X_pred, X = as.data.frame(X_pred), collapse = NULL, ...)

\method{shapviz}{H2OModel}(object, X_pred, X = as.data.frame(X_pred), collapse = NULL, ...)
}
\arguments{
\item{object}{For XGBoost, LightGBM, and H2O, this is the fitted model used to
calculate SHAP values from \code{X_pred}.
In the other cases, it is the object containing the SHAP values.}

\item{...}{Parameters passed to other methods (currently only used by
the \code{predict()} functions of XGBoost, LightGBM, and H2O).}

\item{X}{Matrix or data.frame of feature values used for visualization.
Must contain at least the same column names as the SHAP matrix represented by
\code{object}/\code{X_pred} (after optionally collapsing some of the SHAP columns).}

\item{baseline}{Optional baseline value, representing the average response at the
scale of the SHAP values. It will be used for plot methods that explain single
predictions.}

\item{collapse}{A named list of character vectors. Each vector specifies the
feature names whose SHAP values need to be summed up.
The names determine the resulting collapsed column/dimension names.}

\item{S_inter}{Optional 3D array of SHAP interaction values.
If \code{object} has shape n x p, then \code{S_inter} needs to be of
shape n x p x p. Summation over the second (or third) dimension should yield the
usual SHAP values. Furthermore, dimensions 2 and 3 are expected to be symmetric.
Default is \code{NULL}.}

\item{X_pred}{Data set as expected by the \code{predict()} function of
XGBoost, LightGBM, or H2O. For XGBoost, a matrix or \code{xgb.DMatrix},
for LightGBM a matrix, and for H2O a \code{data.frame} or an \code{H2OFrame}.
Only used for XGBoost, LightGBM, or H2O objects.}

\item{which_class}{In case of a multiclass or multioutput setting,
which class/output (>= 1) to explain. Currently relevant for XGBoost, LightGBM,
and kernelshap.}

\item{interactions}{Should SHAP interactions be calculated (default is \code{FALSE})?
Only available for XGBoost.}
}
\value{
An object of class "shapviz" with the following elements:
\itemize{
\item \code{S}: Numeric matrix of SHAP values.
\item \code{X}: \code{data.frame} containing the feature values corresponding to \code{S}.
\item \code{baseline}: Baseline value, representing the average prediction at the
scale of the SHAP values.
\item \code{S_inter}: Numeric array of SHAP interaction values (or \code{NULL}).
}
}
\description{
This function creates an object of class "shapviz" from a matrix of SHAP values, or
from a fitted model of type
\itemize{
\item XGBoost,
\item LightGBM, or
\item H2O (tree-based regression or binary classification model).
}

Furthermore, \code{\link[=shapviz]{shapviz()}} can digest the results of
\itemize{
\item \code{fastshap::explain()},
\item \code{shapr::explain()},
\item \code{treeshap::treeshap()},
\item \code{DALEX::predict_parts()}, and
\item \code{kernelshap::kernelshap()},
}

check the vignettes for examples.
}
\details{
Together with the main input, a data set \code{X} of feature values is required,
used only for visualization. It can therefore contain character or factor
variables, even if the SHAP values were calculated from a purely numerical feature
matrix. In addition, to improve visualization, it can sometimes be useful to truncate
gross outliers, logarithmize certain columns, or replace missing values with an
explicit value.

SHAP values of dummy variables can be combined using the convenient
\code{collapse} argument.
Multi-output models created from XGBoost, LightGBM, or {kernelshap}
return a "mshapviz" object, containing a "shapviz" object per output.
}
\section{Methods (by class)}{
\itemize{
\item \code{shapviz(default)}: Default method to initialize a "shapviz" object.

\item \code{shapviz(matrix)}: Creates a "shapviz" object from a matrix of SHAP values.

\item \code{shapviz(xgb.Booster)}: Creates a "shapviz" object from an XGBoost model.

\item \code{shapviz(lgb.Booster)}: Creates a "shapviz" object from a LightGBM model.

\item \code{shapviz(explain)}: Creates a "shapviz" object from \code{fastshap::explain()}.

\item \code{shapviz(treeshap)}: Creates a "shapviz" object from \code{treeshap::treeshap()}.

\item \code{shapviz(predict_parts)}: Creates a "shapviz" object from \code{DALEX::predict_parts()}.

\item \code{shapviz(shapr)}: Creates a "shapviz" object from \code{shapr::explain()}.

\item \code{shapviz(kernelshap)}: Creates a "shapviz" object from \code{kernelshap::kernelshap()}.

\item \code{shapviz(H2ORegressionModel)}: Creates a "shapviz" object from a (tree-based) H2O regression model.

\item \code{shapviz(H2OBinomialModel)}: Creates a "shapviz" object from a (tree-based) H2O binary classification model.

\item \code{shapviz(H2OModel)}: Creates a "shapviz" object from a (tree-based) H2O model (base class).

}}
\examples{
S <- matrix(c(1, -1, -1, 1), ncol = 2, dimnames = list(NULL, c("x", "y")))
X <- data.frame(x = c("a", "b"), y = c(100, 10))
shapviz(S, X, baseline = 4)

# XGBoost models
X_pred <- data.matrix(iris[, -1L])
dtrain <- xgboost::xgb.DMatrix(X_pred, label = iris[, 1L])
fit <- xgboost::xgb.train(data = dtrain, nrounds = 50L, nthread = 1L)

# Will use numeric matrix "X_pred" as feature matrix
x <- shapviz(fit, X_pred = X_pred)
x
sv_dependence(x, "Species")

# Will use original values as feature matrix
x <- shapviz(fit, X_pred = X_pred, X = iris)
sv_dependence(x, "Species")

# "X_pred" can also be passed as xgb.DMatrix, but only if X is passed as well!
x <- shapviz(fit, X_pred = dtrain, X = iris)

# Multiclass setting
params <- list(objective = "multi:softprob", num_class = 3L)
X_pred <- data.matrix(iris[, -5L])
dtrain <- xgboost::xgb.DMatrix(X_pred, label = as.integer(iris[, 5L]) - 1L)
fit <- xgboost::xgb.train(
  params = params, data = dtrain, nrounds = 50L, nthread = 1L
)

# Select specific class
x <- shapviz(fit, X_pred = X_pred, which_class = 3L)
x

# Or combine all classes to "mshapviz" object
x <- shapviz(fit, X_pred = X_pred)
x

# What if we would have one-hot-encoded values and want to explain the original column?
X_pred <- stats::model.matrix(~ . -1, iris[, -1L])
dtrain <- xgboost::xgb.DMatrix(X_pred, label = as.integer(iris[, 1L]))
fit <- xgboost::xgb.train(data = dtrain, nrounds = 50L)
x <- shapviz(
  fit,
  X_pred = X_pred,
  X = iris,
  collapse = list(Species = c("Speciessetosa", "Speciesversicolor", "Speciesvirginica"))
)
summary(x)

# Similarly with LightGBM
if (requireNamespace("lightgbm", quietly = TRUE)) {
  fit <- lightgbm::lgb.train(
    params = list(objective = "regression", num_thread = 1L),
    data = lightgbm::lgb.Dataset(X_pred, label = iris[, 1L]),
    nrounds = 50L,
    verbose = -2L
  )

  x <- shapviz(fit, X_pred = X_pred)
  x

  # Multiclass
  params <- list(objective = "multiclass", num_class = 3L, num_thread = 1L)
  X_pred <- data.matrix(iris[, -5L])
  dtrain <- lightgbm::lgb.Dataset(X_pred, label = as.integer(iris[, 5L]) - 1L)
  fit <- lightgbm::lgb.train(params = params, data = dtrain, nrounds = 50L)

  # Select specific class
  x <- shapviz(fit, X_pred = X_pred, which_class = 3L)
  x

  # Or combine all classes to a "mshapviz" object
  mx <- shapviz(fit, X_pred = X_pred)
  mx
  all.equal(mx[[3L]], x)
}
}
\seealso{
\code{\link[=sv_importance]{sv_importance()}}, \code{\link[=sv_dependence]{sv_dependence()}}, \code{\link[=sv_interaction]{sv_interaction()}},
\code{\link[=sv_waterfall]{sv_waterfall()}}, \code{\link[=sv_force]{sv_force()}}, \code{\link[=collapse_shap]{collapse_shap()}}
}
