% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inbinom.R
\name{inbinom}
\alias{inbinom}
\title{Visualization of Random Variate Generation for the Negative Binomial Distribution}
\usage{
inbinom(
  u = runif(1),
  size,
  prob,
  mu,
  minPlotQuantile = 0,
  maxPlotQuantile = 0.95,
  plot = TRUE,
  showCDF = TRUE,
  showPMF = TRUE,
  showECDF = TRUE,
  show = NULL,
  maxInvPlotted = 50,
  plotDelay = 0,
  sampleColor = "red3",
  populationColor = "grey",
  showTitle = TRUE,
  respectLayout = FALSE,
  ...
)
}
\arguments{
\item{u}{vector of uniform(0,1) random numbers, or NULL to show population
figures only}

\item{size}{target for number of successful trials, or dispersion
parameter (the shape parameter of the gamma mixing
distribution).  Must be strictly positive, need not be
integer.}

\item{prob}{Probability of success in each trial;  '0 < prob <= 1'}

\item{mu}{alternative parameterization via mean}

\item{minPlotQuantile}{minimum quantile to plot}

\item{maxPlotQuantile}{maximum quantile to plot}

\item{plot}{logical; if \code{TRUE} (default), one or more plots will appear
(see parameters below); otherwise no plots appear}

\item{showCDF}{logical; if \code{TRUE} (default), cdf plot appears, otherwise cdf 
plot is suppressed}

\item{showPMF}{logical; if \code{TRUE} (default), PMF plot appears,
otherwise PMF plot is suppressed}

\item{showECDF}{logical; if \code{TRUE} (default), ecdf plot appears,
otherwise ecdf plot is suppressed}

\item{show}{octal number (0-7) indicating plots to display;  4: CDF, 2: PMF, 
1: ECDF; sum for desired combination}

\item{maxInvPlotted}{number of inversions to plot across CDF before switching to 
plotting quantiles only}

\item{plotDelay}{delay in seconds between CDF plots}

\item{sampleColor}{Color used to display random sample from distribution}

\item{populationColor}{Color used to display population}

\item{showTitle}{logical; if \code{TRUE} (default), displays a title in the 
first of any displayed plots}

\item{respectLayout}{logical; if \code{TRUE} (default), respects existing 
settings for device layout}

\item{...}{Possible additional arguments. Currently, additional arguments not considered.}
}
\value{
A vector of Negative Binomial random variates
}
\description{
Generates random variates from the Negative Binomial distribution by inversion.
 Optionally graphs the population cumulative distribution function and
 associated random variates, the population probability mass
 function and a histogram of the random variates, and the empirical
 cumulative distribution function versus the population cumulative
 distribution function.
}
\details{
Generates random variates from the Negative Binomial distribution, and optionally,
 illustrates
 \itemize{
   \item the use of the inverse-CDF technique,
   \item the effect of random sampling variability in relation to the PMF and CDF.
 }
 When all of the graphics are requested,
 \itemize{
   \item the first graph illustrates the use of the inverse-CDF technique by
       graphing the population CDF and the transformation of the random numbers
       to random variates,
   \item the second graph illustrates the effect of random sampling variability
       by graphing the population PMF and the histogram associated with the
       random variates, and
   \item the third graph illustrates effect of random sampling variability by
       graphing the population CDF and the empirical CDF associated with the
       random variates.
 }
 All aspects of the random variate generation algorithm are output in red by
 default, which can be changed by specifying \code{sampleColor}.
 All aspects of the population distribution are output in gray by default,
 which can be changed by specifying \code{populationColor}.

The negative binomial distribution with \code{size} \eqn{= n} and
 \code{prob} \eqn{= p} has density

          \deqn{p(x) = \frac{\Gamma(x+n)}{\Gamma(n) \ x!} p^n (1-p)^x}{%
                p(x) = Gamma(x+n)/(Gamma(n) x!) p^n (1-p)^x}

 for \eqn{x = 0, 1, 2, \ldots, n > 0} and \eqn{0 < p \leq 1}. This represents the
 number of failures which occur in a sequence of Bernoulli trials before a
 target number of successes is reached.

 The mean is \eqn{\mu}{μ}\eqn{ = n(1 - p)/p} and variance \eqn{n(1 - p)/p^2}

The algorithm for generating random variates from the negative binomial distribution is
  synchronized (one random variate for each random number) and monotone in u.
  This means that the variates generated here might be useful in some variance
  reduction techniques used in Monte Carlo and discrete-event simulation.

  Values from the u vector are plotted in the cdf plot along the vertical axis
  as colored dots.  A horizontal, dashed, colored line extends from the dot to
  the population cdf.  At the intersection, a vertical, dashed colored line
  extends downward to the horizontal axis, where a second colored dot, denoting
  the associated negative binomial random variate is plotted.

  This is not a particularly fast variate generation algorithm because it uses
  the base R \code{qnbinom} function to invert the values contained in \code{u}.

  All of the elements of the \code{u} vector must be between 0 and 1.
  Alternatively, \code{u} can be \code{NULL} in which case plot(s) of the
  theoretical PMF and cdf are displayed according to plotting parameter
  values (defaulting to display of both the PMF and cdf).

  The \code{show} parameter can be used as a shortcut way to denote plots to
  display.  The argument to \code{show} can be either:
  \itemize{
    \item a binary vector of length three, where the entries from left to right
        correspond to \code{showCDF}, \code{showPMF}, and \code{showECDF},
        respectively.  For each entry, a 1 indicates the plot should be
        displayed, and a 0 indicates the plot should be suppressed.
    \item an integer in [0,7] interpreted similar to the Unix chmod command.  That
        is, the integer's binary representation can be transformed into a
        length-three vector discussed above (e.g., 6 corresponds to c(1,1,0)).
        See examples.
  }
  Any valid value for \code{show} takes precedence over existing individual
  values for \code{showCDF}, \code{showPMF}, and \code{showECDF}.

  If \code{respectLayout} is \code{TRUE}, the function respects existing
  settings for device layout.  Note, however, that if the number of plots
  requested (either via \code{show} or via \code{showCDF}, \code{showPMF}, and
  \code{showECDF}) exceeds the number of plots available in the current layout
  (as determined by \code{prod(par("mfrow"))}), the function will display all
  requested plots but will also display a warning message indicating that the
  current layout does not permit simultaneous viewing of all requested plots.
  The most recent plot with this attribute can be further annotated after the call.

  If \code{respectLayout} is \code{FALSE}, any existing user settings for device
  layout are ignored.  That is, the function uses \code{par} to explicitly set
  \code{mfrow} sufficient to show all requested plots stacked vertically to
  align their horizontal axes, and then resets row, column, and margin settings
  to their prior state on exit.

  The \code{minPlotQuantile} and \code{maxPlotQuantile} arguments are present in
  order to compress the plots horizontally.   The random variates generated are
  not impacted by these two arguments.  Vertical, dotted, black lines are
  plotted at the associated quantiles on the plots.

  \code{plotDelay} can be used to slow down or halt the variate generation for
  classroom explanation.

  In the plot associated with the PMF, the maximum plotting height is
  associated with 125\% of the maximum height of PMF. Any histogram cell
  that extends above this limit will have three dots appearing above it.
}
\examples{
 inbinom(0.5, size = 10, mu = 10)

 set.seed(8675309)
 inbinom(runif(10), 10, 0.25, showPMF = TRUE)

 set.seed(8675309)
 inbinom(runif(10), 10, 0.25, showECDF = TRUE)

 set.seed(8675309)
 inbinom(runif(10), 10, 0.25, showPMF = TRUE, showECDF = TRUE, sampleColor = "blue3")

 set.seed(8675309)
 inbinom(runif(10), 10, 0.25, showPMF = TRUE, showCDF = FALSE)

 \dontrun{
 inbinom(runif(100), 10, 0.25, showPMF = TRUE, minPlotQuantile = 0.02, maxPlotQuantile = 0.98)
 }

 # plot the PMF and CDF without any variates
 inbinom(NULL, 10, 0.25, showPMF = TRUE, showCDF = TRUE)

 # plot CDF with inversion and PMF using show
 \dontrun{
 inbinom(runif(10), 10, 0.25, show = c(1,1,0))
 }
 inbinom(runif(10), 10, 0.25, show = 6)

 # plot CDF with inversion and ECDF using show, using vunif
 \dontrun{
 inbinom(vunif(10), 10, 0.25, show = c(1,0,1))
 }
 inbinom(vunif(10), 10, 0.25, show = 5)

 # plot CDF with inversion, PMF, and ECDF using show
 \dontrun{
 inbinom(vunif(10), 10, 0.25, show = c(1,1,1))
 }
 inbinom(vunif(10), 10, 0.25, show = 7)

 # plot three different CDF+PMF+ECDF horizontal displays,
 # with title only on the first display
 par(mfrow = c(3,3))  # 3 rows, 3 cols, filling rows before columns
 set.seed(8675309)
 inbinom(runif(20), 10, 0.25, show = 7, respectLayout = TRUE)
 inbinom(runif(20), 10, 0.25, show = 7, respectLayout = TRUE, showTitle = FALSE)
 inbinom(runif(20), 10, 0.25, show = 7, respectLayout = TRUE, showTitle = FALSE)

 \dontrun{
 # display animation of all components
 inbinom(runif(10), 10, 0.25, show = 7, plotDelay = 0.1)

 # display animation of CDF and PMF components only
 inbinom(runif(10), 10, 0.25, show = 5, plotDelay = 0.1)

 # pause at each stage of inversion
 inbinom(runif(10), 10, 0.25, show = 7, plotDelay = -1)
 }

}
\seealso{
\code{\link[=rnbinom]{stats::rnbinom}}

\code{\link[=runif]{stats::runif}}, 
         \code{\link[=vunif]{simEd::vunif}}
}
\author{
Barry Lawson (\email{blawson@richmond.edu}), \cr
   Larry Leemis (\email{leemis@math.wm.edu}), \cr
   Vadim Kudlay (\email{vadim.kudlay@richmond.edu})
}
\concept{random variate generation}
\keyword{distribution}
\keyword{dynamic}
\keyword{hplot}
