% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions.R
\name{simulate_kde}
\alias{simulate_kde}
\title{Simulation with Kernel Density Estimation}
\usage{
simulate_kde(
  x,
  n = 100,
  distr = "norm",
  const.only = FALSE,
  seed = NULL,
  parallel = FALSE,
  ...
)
}
\arguments{
\item{x}{a numeric vector, matrix or data frame; data.}

\item{n}{integer; the number of random values will be generated.}

\item{distr}{character; instrumental or candidate distribution name. See details.}

\item{const.only}{logical; if \code{TRUE}, the constant of the Accept-Reject method will be returned.}

\item{seed}{a single value, interpreted as an integer, or \code{NULL} (default).}

\item{parallel}{logical; if \code{TRUE} parallel generator will be worked. \code{FALSE} is default.}

\item{...}{other parameters for functions \code{\link[ks]{kde}}.}
}
\value{
list of given data, simulated values, kernel density estimation and the constant of the Accept-Reject method when \code{const.only} is \code{FALSE} (default).
}
\description{
Generates random values from a univariate and multivariate continuous distribution by using kernel density estimation based on a sample. The function uses the Accept-Reject method.
}
\details{
Such function uses the function \code{\link[ks]{kde}} as kernel density estimator.

The Accept-Reject method is used to simulate random variables.
Following code named distributions can be used as a value of the argument \code{distr} and an instrumental or candidate distribution of the simulation method.
For univariate distributions:
\describe{
\item{norm}{normal distribution (default), \eqn{(-\infty,+\infty)}}
\item{cauchy}{Cauchy distribution, \eqn{(-\infty,+\infty)}}
\item{lnorm}{log-normal distribution, \eqn{(0,+\infty)}}
\item{exp}{exponential distribution, \eqn{(0,+\infty)}}
\item{gamma}{gamma distribution, \eqn{(0,+\infty)}}
\item{weibull}{Weibull distribution, \eqn{(0,+\infty)}}
\item{unif}{uniform distribution, \eqn{(a,b)}}
}
And you can choose the best fitting instrumental distribution to simulate random variables more effectively by using \code{\link{find_best_fit}}. See examples.

For multivariate distributions, "norm" (multivariate normal distribution) is used.
}
\examples{
## 1-dimensional data
data(faithful)
hist(faithful$eruptions)
res <- simukde::simulate_kde(x = faithful$eruptions, n = 100, parallel = FALSE)
hist(res$random.values)

## Simulation with the best fitting instrumental distribution
data(faithful)
par(mfrow = c(1, 3))
hist(faithful$eruptions)
fit <- simukde::find_best_fit(x = faithful$eruptions, positive = TRUE)
res <- simukde::simulate_kde(
  x = faithful$eruptions, n = 100,
  distr = fit$distribution, parallel = FALSE
)
hist(res$random.values)
par(mfrow = c(1, 1))
\donttest{
## 2-dimensional data
data(faithful)
res <- simukde::simulate_kde(x = faithful, n = 100)
plot(res$kde, display = "filled.contour2")
points(x = res$random.values, cex = 0.25, pch = 16, col = "green")
points(x = faithful, cex = 0.25, pch = 16, col = "black")}
\dontshow{
## 2-dimensional data
data(faithful)
res <- simukde::simulate_kde(x = faithful, n = 1, parallel = FALSE)
plot(res$kde, display = "filled.contour2")
points(x = res$random.values, cex = 0.5, pch = 16, col = "blue")
points(x = faithful, cex = 0.25, pch = 16, col = "gray")}
}
\references{
\itemize{
 \item Tarn Duong (2018). ks: Kernel Smoothing. R package version 1.11.2. \url{https://CRAN.R-project.org/package=ks}
 \item Christian P. Robert and George Casella (2010) Introducing Monte Carlo Methods with R. Springer. Pages 51-57.
}
}
\seealso{
\code{\link{find_best_fit}}, \code{\link[ks]{kde}}
}
