% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sjPlotInteractions.R
\name{sjp.int}
\alias{sjp.int}
\title{Plot interaction effects of (generalized) linear (mixed) models}
\usage{
sjp.int(fit, type = "cond", int.term = NULL, int.plot.index = NULL,
  diff = FALSE, moderatorValues = "minmax", swapPredictors = FALSE,
  plevel = 0.05, title = NULL, fillColor = "grey", fillAlpha = 0.3,
  geom.colors = "Set1", geom.size = NULL, axisTitle.x = NULL,
  axisTitle.y = NULL, axisLabels.x = NULL, legendTitle = NULL,
  legendLabels = NULL, showValueLabels = FALSE, breakTitleAt = 50,
  breakLegendLabelsAt = 20, breakLegendTitleAt = 20, axisLimits.x = NULL,
  axisLimits.y = NULL, y.offset = 0.07, gridBreaksAt = NULL,
  showCI = FALSE, valueLabel.digits = 2, facet.grid = FALSE,
  printPlot = TRUE)
}
\arguments{
\item{fit}{the fitted (generalized) linear (mixed) model object, including interaction terms. Accepted model
classes are
\itemize{
  \item linear models (\code{\link{lm}})
  \item generalized linear models (\code{\link{glm}})
  \item linear mixed effects models (\code{\link[lme4]{lmer}})
  \item generalized linear mixed effects models (\code{\link[lme4]{glmer}})
  \item non-linear mixed effects models (\code{\link[lme4]{nlmer}})
  \item linear mixed effects models (\code{\link[nlme]{lme}}, but only for \code{type = "eff"})
  \item generalized least squares models (\code{\link[nlme]{gls}}, but only for \code{type = "eff"})
  \item panel data estimators (\code{plm})
}}

\item{type}{interaction plot type. Use one of following values:
\describe{
  \item{\code{type = "cond"}}{(default) plots the mere \emph{change} of the moderating effect on the response value (conditional effect). See 'Details'.}
  \item{\code{type = "eff"}}{plots the overall moderation effect on the response value. See 'Details'.}
  \item{\code{type = "emm"}}{plots the estimated marginal means (least square means). If this type is chosen, not all function arguments are applicable. See 'Details'.}
}}

\item{int.term}{select interaction term of \code{fit} (as character), which should be plotted
when using \code{type = "eff"}. By default, this argument can be ignored
(i.e. \code{int.term = NULL}). See 'Details'.}

\item{int.plot.index}{numeric vector with index numbers that indicate which 
interaction terms should be plotted in case the \code{fit} has more than
one interaction. By default, this values is \code{NULL}, hence all interactions
are plotted.}

\item{diff}{if \code{FALSE} (default), the minimum and maximum interaction effects of the moderating variable
is shown (one line each). if \code{TRUE}, only the difference between minimum and maximum interaction effect
is shown (single line). Only applies to \code{type = "cond"}.}

\item{moderatorValues}{indicates which values of the moderator variable should be used when plotting the effects of the
independent variable on the dependent variable.
\describe{
  \item{\code{"minmax"}}{(default) minimum and maximum values (lower and upper bounds) of the moderator are used to plot the interaction between independent variable and moderator.}
  \item{\code{"meansd"}}{uses the mean value of the moderator as well as one standard deviation below and above mean value to plot the effect of the moderator on the independent variable (following the convention suggested by Cohen and Cohen and popularized by Aiken and West, i.e. using the mean, the value one standard deviation above, and the value one standard deviation below the mean as values of the moderator, see \href{http://www.theanalysisfactor.com/3-tips-interpreting-moderation/}{Grace-Martin K: 3 Tips to Make Interpreting Moderation Effects Easier}).}
  \item{\code{"zeromax"}}{is similar to the \code{"minmax"} option, however, \code{0} is always used as minimum value for the moderator. This may be useful for predictors that don't have an empirical zero-value, but absence of moderation should be simulated by using 0 as minimum.}
  \item{\code{"quart"}}{calculates and uses the quartiles (lower, median and upper) of the moderator value.}
}}

\item{swapPredictors}{if \code{TRUE}, the predictor on the x-axis and the moderator value in an interaction are
swapped. For \code{type = "eff"}, the first interaction term is used as moderator and the second term
is plotted at the x-axis. For \code{type = "cond"}, the interaction's predictor with less unique values is 
printed along the x-axis. Default is \code{FALSE}, so the second predictor in an interaction, respectively 
the predictor with more unique values is printed along the x-axis.}

\item{plevel}{indicates at which p-value an interaction term is considered as \emph{significant},
i.e. at which p-level an interaction term will be considered for plotting. Default is
0.05 (5 percent), hence, non-significant interactions are excluded by default. This
argument does not apply to \code{type = "eff"}.}

\item{title}{a default title used for the plots. Should be a character vector
of same length as interaction plots to be plotted. Default value is \code{NULL}, which means that each plot's title
includes the dependent variable as well as the names of the interaction terms.}

\item{fillColor}{fill color of the shaded area between the minimum and maximum lines. Default is \code{"grey"}.
Either set \code{fillColor} to \code{NULL} or use 0 for \code{fillAlpha} if you want to hide the shaded area.}

\item{fillAlpha}{alpha value (transparancy) of the shaded area between the minimum and maximum lines. Default is 0.4.
Use either 0 or set \code{fillColor} to \code{NULL} if you want to hide the shaded area.}

\item{geom.colors}{vector of color values. First value is the color of the line indicating the lower bound of
the interaction term (moderator value). Second value is the color of the line indicating the upper bound of
the interaction term (moderator value). Third value, if applicable, is the color of the line indicating the
mean value of the interaction term (moderator value). Third value is only used when 
\code{moderatorValues = "meansd"}. Or, if \code{diff = TRUE}, only one color value for the 
line indicating the upper difference between lower and upper bound of interaction terms.}

\item{geom.size}{size resp. width of the geoms (bar width, line thickness or point size, depending on \code{type} argument).
Note that bar and bin widths mostly need smaller values than dot sizes (i.e. if \code{type = "dots"}).
By default, \code{geom.size = NULL}, which means that this argument is automatically
adjusted depending on the plot type.}

\item{axisTitle.x}{a default title used for the x-axis. Should be a character vector
of same length as interaction plots to be plotted. Default value is \code{NULL},
which means that each plot's x-axis uses the predictor's name as title.}

\item{axisTitle.y}{a default title used for the y-axis. Default value is \code{NULL},
which means that each plot's y-axis uses the dependent variable's name as title.}

\item{axisLabels.x}{character vector with value labels of the repeated measure variable
that are used for labelling the x-axis.}

\item{legendTitle}{title of the diagram's legend. A character vector of same length as 
amount of interaction plots to be plotted (i.e. one vector element for each
plot's legend title).}

\item{legendLabels}{labels for the guide/legend. Either a character vector of same length as
amount of legend labels of the plot, or a \code{list} of character vectors, if more than one
interaction plot is plotted (i.e. one vector of legend labels for each interaction plot).
Default is \code{NULL}, so the name of the predictor with min/max-effect is used 
as legend label.}

\item{showValueLabels}{if \code{TRUE}, value labels are plotted along the lines. Default is \code{FALSE}.}

\item{breakTitleAt}{determines how many chars of the plot title are displayed in
one line and when a line break is inserted into the title.}

\item{breakLegendLabelsAt}{determines how many chars of the legend labels are 
displayed in one line and when a line break is inserted.}

\item{breakLegendTitleAt}{determines how many chars of the legend's title 
are displayed in one line and when a line break is inserted.}

\item{axisLimits.x}{numeric vector of length two, defining lower and upper axis limits}

\item{axisLimits.y}{numeric vector of length two, defining lower and upper axis limits
of the y scale. By default, this argument is set to \code{NULL}, i.e. the 
y-axis fits to the required range of the data.}

\item{y.offset}{numeric, offset for text labels when their alignment is adjusted 
to the top/bottom of the geom (see \code{hjust} and \code{vjust}).}

\item{gridBreaksAt}{set breaks for the axis, i.e. at every \code{gridBreaksAt}'th 
position a major grid is being printed.}

\item{showCI}{may be a numeric or logical value. If \code{showCI} is logical and 
\code{TRUE}, a 95\% confidence region will be plotted. If \code{showCI}
if numeric, must be a number between 0 and 1, indicating the proportion
for the confidence regeion (e.g. \code{showCI = 0.9} plots a 90\% CI).
Only applies to \code{type = "emm"} or \code{type = "eff"}.}

\item{valueLabel.digits}{the amount of digits of the displayed value labels. Defaults to 2.}

\item{facet.grid}{\code{TRUE} for faceted plots instead of an integrated single plot.}

\item{printPlot}{logical, if \code{TRUE} (default), plots the results as graph. Use \code{FALSE} if you don't
want to plot any graphs. In either case, the ggplot-object will be returned as value.}
}
\value{
(Insisibily) returns the ggplot-objects with the complete plot-list (\code{plot.list})
          as well as the data frames that were used for setting up the ggplot-objects (\code{data.list}).
}
\description{
Plot regression (predicted values) or probability lines (predicted probabilities) of 
               significant interaction terms to better understand effects
               of moderations in regression models. This function accepts following fitted model classes:
               \itemize{
                 \item linear models (\code{\link{lm}})
                 \item generalized linear models (\code{\link{glm}})
                 \item linear mixed effects models (\code{\link[lme4]{lmer}})
                 \item generalized linear mixed effects models (\code{\link[lme4]{glmer}})
                 \item non-linear mixed effects models (\code{\link[lme4]{nlmer}})
                 \item linear mixed effects models (\code{\link[nlme]{lme}}, but only for \code{type = "eff"})
                 \item generalized least squares models (\code{\link[nlme]{gls}}, but only for \code{type = "eff"})
                 \item panel data estimators (\code{plm})
               }
               Note that beside interaction terms, also the single predictors of each interaction (main effects)
               must be included in the fitted model as well. Thus, \code{lm(dep ~ pred1 * pred2)} will work, 
               but \code{lm(dep ~ pred1:pred2)} won't!
}
\details{
\describe{
           \item{\code{type = "cond"}}{plots the effective \emph{change} or \emph{impact} 
             (conditional effect) on a dependent variable of a moderation effect, as 
             described in \href{http://www.theanalysisfactor.com/clarifications-on-interpreting-interactions-in-regression/}{Grace-Martin},
             i.e. the difference of the moderation effect on the dependent variable in \emph{presence}
             and \emph{absence} of the moderating effect (\emph{simple slope} plot or 
             \emph{conditional effect}, see \href{http://imaging.mrc-cbu.cam.ac.uk/statswiki/FAQ/SobelTest?action=AttachFile&do=get&target=process.pdf}{Hayes 2012}).
             Hence, this plot type may be used especially for \emph{binary or dummy coded} 
             moderator values (see also \href{http://jee3.web.rice.edu/interaction-overconfidence.pdf}{Esarey and Summer 2015}).
             This type \emph{does not} show the overall effect (marginal mean, i.e. adjusted
             for all other predictors and covariates) of interactions on the result of Y. Use 
             \code{type = "eff"} for effect displays similar to the \code{\link[effects]{effect}}-function 
             from the \pkg{effects}-package.
           }
           \item{\code{type = "eff"}}{plots the overall effects (marginal effects) of the interaction, with all remaining
             covariates set to the mean. Effects are calculated using the \code{\link[effects]{effect}}-
             function from the \pkg{effects}-package. \cr \cr
             Following arguments \emph{do not} apply to this function: \code{diff}, \code{axisLabels.x}.
           }
           \item{\code{type = "emm"}}{plots the estimated marginal means of repeated measures designs,
             like two-way repeated measures AN(C)OVA. In detail, this type plots estimated marginal means 
             (also called \emph{least square means} or \emph{marginal means}) of (significant) interaction terms.
             The fitted models may be linear (mixed effects) 
             models of class \code{\link{lm}} or \code{\link[lme4]{merMod}}. This function may be used, for example,
             to plot differences in interventions between control and treatment groups over multiple time points.
             \itemize{
               \item Following paramters apply to this plot type: \code{showCI}, \code{valueLabel.digits} and \code{axisLabels.x}.
               \item Following arguments \emph{do not} apply to this function: \code{int.term}, \code{int.plot.index}, \code{diff}, \code{moderatorValues}, \code{fillColor}, \code{fillAlpha}.
             }
           }
         }
         The argument \code{int.term} only applies to \code{type = "eff"} and can be used
         to select a specific interaction term of the model that should be plotted. The function
         then calls \code{effect(int.term, fit)} to compute effects for this specific interaction
         term only. This approach is recommended, when the fitted model contains many observations
         and/or variables, which may slow down the effect-computation dramatically. In such cases,
         consider computing effects for selected interaction terms only with \code{int.terms}.
         See 'Examples'.
}
\note{
Note that beside interaction terms, also the single predictors of each interaction (main effects)
       must be included in the fitted model as well. Thus, \code{lm(dep ~ pred1 * pred2)} will work, 
       but \code{lm(dep ~ pred1:pred2)} won't! \cr \cr
       For \code{type = "emm"}, all interaction terms have to be \code{\link{factor}}s!
       Furthermore, for \code{type = "eff"}, predictors of interactions that are introduced first into the model
       are used as grouping variable, while the latter predictor is printed along the x-axis
       (i.e. lm(y~a+b+a:b) means that "a" is used as grouping variable and "b" is plotted along the x-axis).
}
\examples{
# Note that the data sets used in this example may not be perfectly suitable for
# fitting linear models. I just used them because they are part of the R-software.

# fit "dummy" model. Note that moderator should enter
# first the model, followed by predictor. Else, use
# argument "swapPredictors" to change predictor on 
# x-axis with moderator
fit <- lm(weight ~ Diet * Time, data = ChickWeight)

# show summary to see significant interactions
summary(fit)

# plot regression line of interaction terms, including value labels
sjp.int(fit, type = "eff", showValueLabels = TRUE)


# load sample data set
library(sjmisc)
data(efc)
# create data frame with variables that should be included
# in the model
mydf <- data.frame(usage = efc$tot_sc_e,
                   sex = efc$c161sex,
                   education = efc$c172code,
                   burden = efc$neg_c_7,
                   dependency = efc$e42dep)
# convert gender predictor to factor
mydf$sex <- relevel(factor(mydf$sex), ref = "2")
# fit "dummy" model
fit <- lm(usage ~ .*., data = mydf)
summary(fit)

# plot interactions. note that type = "cond" only considers 
# significant interactions by default. use "plevel" to 
# adjust p-level sensivity
sjp.int(fit, type = "cond")

# plot only selected interaction term for
# type = "eff"
sjp.int(fit, type = "eff", int.term = "sex*education")

# plot interactions, using mean and sd as moderator
# values to calculate interaction effect
sjp.int(fit, type = "eff", moderatorValues = "meansd")
sjp.int(fit, type = "cond", moderatorValues = "meansd")

# plot interactions, including those with p-value up to 0.1
sjp.int(fit,
        type = "cond",
        plevel = 0.1)

# -------------------------------
# Predictors for negative impact of care.
# Data from the EUROFAMCARE sample dataset
# -------------------------------
library(sjmisc)
data(efc)
# create binary response
y <- ifelse(efc$neg_c_7 < median(stats::na.omit(efc$neg_c_7)), 0, 1)
# create data frame for fitted model
mydf <- data.frame(y = as.factor(y),
                   sex = as.factor(efc$c161sex),
                   barthel = as.numeric(efc$barthtot))
# fit model
fit <- glm(y ~ sex * barthel,
           data = mydf,
           family = binomial(link = "logit"))
# plot interaction, increase p-level sensivity
sjp.int(fit,
        type = "eff",
        legendLabels = get_labels(efc$c161sex),
        plevel = 0.1)

sjp.int(fit,
        type = "cond",
        legendLabels = get_labels(efc$c161sex),
        plevel = 0.1)
        
\dontrun{
# -------------------------------
# Plot estimated marginal means
# -------------------------------
# load sample data set
library(sjmisc)
data(efc)
# create data frame with variables that should be included
# in the model
mydf <- data.frame(burden = efc$neg_c_7,
                   sex = efc$c161sex,
                   education = efc$c172code)
# convert gender predictor to factor
mydf$sex <- factor(mydf$sex)
mydf$education <- factor(mydf$education)
# name factor levels and dependent variable
levels(mydf$sex) <- c("female", "male")
levels(mydf$education) <- c("low", "mid", "high")
mydf$burden <- set_label(mydf$burden, "care burden")
# fit "dummy" model
fit <- lm(burden ~ .*., data = mydf)
summary(fit)

# plot marginal means of interactions, no interaction found
sjp.int(fit, type = "emm")
# plot marginal means of interactions, including those with p-value up to 1
sjp.int(fit, type = "emm", plevel = 1)
# swap predictors
sjp.int(fit, 
        type = "emm",
        plevel = 1, 
        swapPredictors = TRUE)

# -------------------------------
# Plot effects
# -------------------------------
# add continuous variable
mydf$barthel <- efc$barthtot
# re-fit model with continuous variable
fit <- lm(burden ~ .*., data = mydf)

# plot effects
sjp.int(fit, type = "eff", showCI = TRUE)

# plot effects, faceted
sjp.int(fit, 
        type = "eff", 
        int.plot.index = 3,
        showCI = TRUE,
        facet.grid = TRUE)}

}
\references{
\itemize{
             \item Aiken and West (1991). Multiple Regression: Testing and Interpreting Interactions.
             \item Brambor T, Clark WR and Golder M (2006) Understanding Interaction Models: Improving Empirical Analyses. Political Analysis 14: 63-82 \href{https://files.nyu.edu/mrg217/public/pa_final.pdf}{download}
             \item Esarey J, Sumner JL (2015) Marginal Effects in Interaction Models: Determining and Controlling the False Positive Rate. \href{http://jee3.web.rice.edu/interaction-overconfidence.pdf}{download}
             \item Fox J (2003) Effect displays in R for generalised linear models. Journal of Statistical Software 8:15, 1–27, \href{http://www.jstatsoft.org/v08/i15/}{<http://www.jstatsoft.org/v08/i15/>}
             \item Hayes AF (2012) PROCESS: A versatile computational tool for observed variable mediation, moderation, and conditional process modeling [White paper] \href{http://imaging.mrc-cbu.cam.ac.uk/statswiki/FAQ/SobelTest?action=AttachFile&do=get&target=process.pdf}{download}
             \item \href{http://www.theanalysisfactor.com/interpreting-interactions-in-regression/}{Grace-Martin K: Interpreting Interactions in Regression}
             \item \href{http://www.theanalysisfactor.com/clarifications-on-interpreting-interactions-in-regression/}{Grace-Martin K: Clarifications on Interpreting Interactions in Regression}
             \item \href{http://www.theanalysisfactor.com/3-tips-interpreting-moderation/}{Grace-Martin K: 3 Tips to Make Interpreting Moderation Effects Easier}
             \item \href{http://www.theanalysisfactor.com/using-adjusted-means-to-interpret-moderators-in-analysis-of-covariance/}{Grace-Martin K: Using Adjusted Means to Interpret Moderators in Analysis of Covariance.}
             }
}
\seealso{
\href{http://www.strengejacke.de/sjPlot/sjp.int/}{sjPlot manual: sjp.int}
}

