% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/loudness.R
\name{getLoudness}
\alias{getLoudness}
\title{Get loudness}
\usage{
getLoudness(
  x,
  samplingRate = NULL,
  scale = NULL,
  from = NULL,
  to = NULL,
  windowLength = 50,
  step = NULL,
  overlap = 50,
  SPL_measured = 70,
  Pref = 2e-05,
  spreadSpectrum = TRUE,
  summaryFun = c("mean", "median", "sd"),
  reportEvery = NULL,
  cores = 1,
  plot = TRUE,
  savePlots = NULL,
  main = NULL,
  ylim = NULL,
  width = 900,
  height = 500,
  units = "px",
  res = NA,
  mar = c(5.1, 4.1, 4.1, 4.1),
  ...
)
}
\arguments{
\item{x}{path to a folder, one or more wav or mp3 files c('file1.wav',
'file2.mp3'), Wave object, numeric vector, or a list of Wave objects or
numeric vectors}

\item{samplingRate}{sampling rate of \code{x} (only needed if \code{x} is a
numeric vector)}

\item{scale}{maximum possible amplitude of input used for normalization of
input vector (only needed if \code{x} is a numeric vector)}

\item{from, to}{if NULL (default), analyzes the whole sound, otherwise
from...to (s)}

\item{windowLength}{length of FFT window, ms (multiple values in a vector
produce a multi-resolution spectrogram)}

\item{step}{you can override \code{overlap} by specifying FFT step, ms - a
vector of the same length as windowLength (NB: because digital audio is
sampled at discrete time intervals of 1/samplingRate, the actual step and
thus the time stamps of STFT frames may be slightly different, eg 24.98866
instead of 25.0 ms)}

\item{overlap}{overlap between successive FFT frames, \%}

\item{SPL_measured}{sound pressure level at which the sound is presented, dB}

\item{Pref}{reference pressure, Pa (currently has no effect on the estimate)}

\item{spreadSpectrum}{if TRUE, applies a spreading function to account for
frequency masking}

\item{summaryFun}{functions used to summarize each acoustic characteristic,
eg "c('mean', 'sd')"; user-defined functions are fine (see examples); NAs
are omitted automatically for mean/median/sd/min/max/range/sum, otherwise
take care of NAs yourself}

\item{reportEvery}{when processing multiple inputs, report estimated time
left every ... iterations (NULL = default, NA = don't report)}

\item{cores}{number of cores for parallel processing}

\item{plot}{should a spectrogram be plotted? TRUE / FALSE}

\item{savePlots}{full path to the folder in which to save the plots (NULL =
don't save, '' = same folder as audio)}

\item{main}{plot title}

\item{ylim}{frequency range to plot, kHz (defaults to 0 to Nyquist
frequency). NB: still in kHz, even if yScale = bark, mel, or ERB}

\item{width, height, units, res}{graphical parameters for saving plots passed to
\code{\link[grDevices]{png}}}

\item{mar}{margins of the spectrogram}

\item{...}{other plotting parameters passed to \code{\link{spectrogram}}}
}
\value{
Returns a list: \describe{ \item{specSone}{spectrum in bark-sone (one
  per file): a matrix of loudness values in sone, with frequency on the bark
  scale in rows and time (STFT frames) in columns} \item{loudness}{a vector
  of loudness in sone per STFT frame (one per file)} \item{summary}{a
  dataframe of summary loudness measures (one row per file)} }
}
\description{
Estimates subjective loudness per frame, in sone. Based on EMBSD speech
quality measure, particularly the matlab code in Yang (1999) and Timoney et
al. (2004). Note that there are many ways to estimate loudness and many other
factors, ignored by this model, that could influence subjectively experienced
loudness. Please treat the output with a healthy dose of skepticism! Also
note that the absolute value of calculated loudness critically depends on the
chosen "measured" sound pressure level (SPL). \code{getLoudness} estimates
how loud a sound will be experienced if it is played back at an SPL of
\code{SPL_measured} dB. The most meaningful way to use the output is to
compare the loudness of several sounds analyzed with identical settings or of
different segments within the same recording.
}
\details{
Algorithm: calibrates the sound to the desired SPL (Timoney et al., 2004),
extracts a spectrogram with \code{\link[tuneR]{powspec}}, converts to bark
scale with (\code{\link[tuneR]{audspec}}), spreads the spectrum to account
for frequency masking across the critical bands (Yang, 1999), converts dB to
phon by using standard equal loudness curves (ISO 226), converts phon to sone
(Timoney et al., 2004), sums across all critical bands, and applies a
correction coefficient to standardize output. Calibrated so as to return a
loudness of 1 sone for a 1 kHz pure tone with SPL of 40 dB.
}
\examples{
sounds = list(
  white_noise = runif(8000, -1, 1),
  white_noise2 = runif(8000, -1, 1) / 2,  # ~6 dB quieter
  pure_tone_1KHz = sin(2*pi*1000/16000*(1:8000))  # pure tone at 1 kHz
)
l = getLoudness(
    x = sounds, samplingRate = 16000, scale = 1,
    windowLength = 20, step = NULL,
    overlap = 50, SPL_measured = 40,
    Pref = 2e-5, plot = FALSE)
l$summary
# white noise (sound 1) is twice as loud as pure tone at 1 KHz (sound 3),
# and note that the same white noise with lower amplitude has lower loudness
# (provided that "scale" is specified)
# compare: lapply(sounds, range)

\dontrun{
s = soundgen()
# playme(s)
l1 = getLoudness(s, samplingRate = 16000, SPL_measured = 70)
l1$summary
# The estimated loudness in sone depends on target SPL
l2 = getLoudness(s, samplingRate = 16000, SPL_measured = 40)
l2$summary

# ...but not (much) on windowLength and samplingRate
l3 = getLoudness(s, samplingRate = 16000, SPL_measured = 40, windowLength = 50)
l3$summary

# input can be an audio file...
getLoudness('~/Downloads/temp/032_ut_anger_30-m-roar-curse.wav')

...or a folder with multiple audio files
getLoudness('~/Downloads/temp2', plot = FALSE)$summary
# Compare:
analyze('~/Downloads/temp2', pitchMethods = NULL,
        plot = FALSE, silence = 0)$summary$loudness_mean
# (per STFT frame; should be similar if silence = 0, because
# otherwise analyze() discards frames considered silent)

# custom summaryFun
ran = function(x) diff(range(x))
getLoudness('~/Downloads/temp2', plot = FALSE,
            summaryFun = c('mean', 'ran'))$summary
}
}
\references{
\itemize{
  \item ISO 226 as implemented by Jeff Tackett (2005) on
  https://www.mathworks.com/matlabcentral/fileexchange/
  7028-iso-226-equal-loudness-level-contour-signal \item Timoney, J.,
  Lysaght, T., Schoenwiesner, M., & MacManus, L. (2004). Implementing
  loudness models in matlab. \item Yang, W. (1999). Enhanced Modified Bark
  Spectral Distortion (EMBSD): An Objective Speech Quality Measure Based on
  Audible Distortion and Cognitive Model. Temple University. }
}
\seealso{
\code{\link{getRMS}} \code{\link{analyze}}
}
