\name{2. Phenomenological models}
\alias{sraAutoreg}
\alias{sraAutoregLog}
\alias{sraAutoregHerit}
\alias{sraAutoregEvolv}
\title{ Descriptive models of artificial-selection responses: auto-regressive models }
\description{
The \code{sraAutoreg} functions are wrappers for the maximum-likelihood optimization function \code{\link{mle}}. They propose descriptive models for the dynamics of genetic architectures of different complexities based on an auto-regressive framework, additional parameters corresponding to different generation lags. The model can also be fit considering logatithmic, "heritability" and "evolvability" scales.
}

\usage{
sraAutoreg(sradata, active = c(FALSE, TRUE, FALSE, FALSE), 
	start = NULL, fixed = NULL, negative.k = FALSE, 
	rand = 0, rep = 1, ...)
sraAutoregLog(sradata, active = c(FALSE, TRUE, FALSE, FALSE), 
	start = NULL, fixed = NULL, negative.k = FALSE, 
	rand = 0, rep = 1, ...)
sraAutoregHerit(sradata, active = c(FALSE, TRUE, FALSE, FALSE), 
	start = NULL, fixed = NULL, negative.k = FALSE, 
	rand = 0, rep = 1, ...)
sraAutoregEvolv(sradata, active = c(FALSE, TRUE, FALSE, FALSE), 
	start = NULL, fixed = NULL, negative.k = FALSE, 
	rand = 0, rep = 1, ...)
}

\arguments{
  \item{sradata}{ A data object generated by \code{\link{sraData}}. }
  \item{active}{ A vector of four booleans, corresponding to the active lags for both genetic and environmental variances (see Details). By default, only lag 1 is active, corresponding to an exponential change of the variances.}
  \item{start}{ A named list of starting values for the convergence algorithm. \code{NULL} is allowed and lets the function \code{\link{sraStartingvalues}} calculating a (hopefully) educated guess on the starting values. See Details. }
  \item{fixed}{ A named list of the parameters that have to be kept constant. \code{NULL} is allowed. }
  \item{negative.k}{ Whether or not the k parameters can take negative values. Negtive values for k lead to more complex likelihood functions, and the resulting dynamics may display cyclic patterns.}
  \item{rand}{ Amount of randomness for the starting values. Useful in case of convergence issues. Although this variable can take any positive value, reasonable figures should not exceed 0.2. }
  \item{rep}{ Number of convergence attempts. When the likelihood function is complex, which is often the case when the number of parameters exceeds 6 to 8, convergence may often fail or end up on a local maximum. When \code{rep} > 1, several attempts are made from different starting values (the amount of randomness for the starting values being set by the \code{rand} parameter), and the best fit (highest likelihood) is returned. Setting high values of \code{rep} may slow down significantly the convergence process. In practice, 10 to 30 repetitions with \code{rand} = 0.1  are generally enough to ensure convergence. }
  \item{\dots}{ Additional parameters to be passed to \code{\link{mle}}, and thus to \code{\link{optim}}. }
}
\details{
  \strong{Model}
  
  The following summarizes the models developed in Le Rouzic et al. 2010. 
  
  The mean of the population \eqn{\mu}{mu} changes according to the Lande equation (Lande \& Arnold 1983):
  
  \deqn{\mu(t+1) = \mu(t) + VarA(t)*\beta(t),}

  where \eqn{\beta(t)} is the selection gradient at generation t.

  The genetic architecture models predict the dynamics of a parameter P as:
  
  \deqn{P(t+1) = k0 + k1*P(t) + k2*P(t-1) + k3*P(t-2)}
  
  Models with time lags > 3 could be easily implemented, but convergence issues increase with the number of parameters. The first time points are calculated as if \eqn{P(t<1)=P(1)}, e.g. \eqn{P(3) = k0 + k1*P(2) + k2*P(1) + k3*P(1)}. 
  
  Each model considers the dynamics of two independent parameters, one related to the additive genetic variance (\eqn{VarA}), one related to the residual (environmental) variance \eqn{VarE} (which actually also accounts for all non-additive genetic variance). 
  
  The default model \code{sraAutoreg} considers directly the dynamics of \eqn{VarA} (parameters: \code{kA0}, \code{kA1}, \code{kA2}, and \code{kA3}) and the dynamics of \eqn{VarE} (parameters \code{kE0}, \code{kE1}, \code{kE2}, and \code{kE3}).
  
  The log scale turns a multiplicative trait into an additive one, and is particularly relevant for ratio-scale traits (i.e. most quantitative traits such as size, fertility, etc). The original data is transformed assuming log-normality, and the likelihood is computed based on the log-normal density function. 
  
  The "heritability" model \code{sraAutoregHerit} focuses on the dynamics of \eqn{h^2=VarA/(VarA+VarE)}{h2=VarA/(VarA+VarE)}, described by the parameters \code{kA0}, \code{kA1}, \code{kA2}, and \code{kA3}, and considers that \code{kE0}, \code{kE1}, \code{kE2}, and \code{kE3} describe the dynamics of the phenotypic variance \eqn{VarP=VarA+VarE}. Therefore, \eqn{VarE} is constrained both by the dynamics of \eqn{VarP} and the independent dynamics of \eqn{h^2}{h2}. 
  
  The "evolvability" model considers that \code{kA0}, \code{kA1}, \code{kA2}, and \code{kA3} describe the dynamics of \eqn{IA(t) = VarA(t)/(\mu(t)^2)}{IA(t) = VarA(t)/(mu(t)^2)}, and \code{kE0}, \code{kE1}, \code{kE2}, and \code{kE3} the dynamics of \eqn{IE(t) = VarE(t)/(\mu(t)^2)}{IE(t) = VarE(t)/(mu(t)^2)}. 
  
  \strong{Shortcut for active and inactive parameters}
  
  The user will often have to fit models of different complexity. This can be achieve by manipulating the \code{active} vector. \code{c(FALSE,FALSE,FALSE,FALSE)} corresponds to a constant-variance model (no dynamic parameter), \code{c(TRUE,FALSE,FALSE,FALSE)} to a case in which only \code{kA0} and \code{kE0} are active, \code{c(TRUE,TRUE,FALSE,FALSE)} to active parameters for lags 0 and 1 only, etc. The total number of parameters in the model will be \eqn{3+2*x}, where \eqn{x} is the number of \code{TRUE} in the vector \code{active}.
  
  To bypass the constrains of this shortcut, it is possible to specify the active and inactive parameters through the list of starting values. A combination such as \code{active=c(TRUE,FALSE,TRUE,}\code{FALSE)}, \code{start=list(kA1=0,kE3=NA)}, \code{fixed=list(kE2=1)} will lead to a model with 8 active parameters (\code{mu0}, \code{varA0}, \code{varE0}, \code{kA0}, \code{kE0}, \code{kA1} (which starting value will be 0), \code{kA2}, and \code{kE3} (which starting value, specified as \code{NA}, will be determined via the function \code{\link{sraStartingvalues}}. All other parameters are fixed. 
  
  \strong{Parameterization}
  
  The models thus involve up to 11 parameters: three initial values (\eqn{\mu(1)}{mu(1)}, \eqn{VarA(1)} and \eqn{VarE(1)}), four parameters to describe the dynamics of the additive variance (or relative variable such as \eqn{IA} or \eqn{h^2}{h2}) (\code{kA0}, \code{kA1}, \code{kA2}, and \code{kA3}), and four parameters for the environmental variance (or \eqn{IE}, or \eqn{h^2}{h2}): \code{kE0}, \code{kE1}, \code{kE2}, and \code{kE3}. To make numerical convergence more efficient, the following parameterization was implemented: parameters \code{mu0}, \code{logvarA0} and \code{logvarE0} correspond to the estimates of the initial values of, respectively, the population mean, the logarithm of the additive variance, and the logarithm of the environmental variance. The parameters \code{kA0} and \code{kE0} are calculated as relative to the initial values of the dynamic variable, e.g. \code{relativekA0} = \eqn{k0A/VarA(1)} (so that \code{relativekA0} has the same unit and the same order of magnitude as \code{kA1}, \code{kA2} and \code{kA3}).
}

\value{
  The functions return objects of class \code{srafit}, a list containing information about the model, the data, and the parameter estimates. Some standard \R functions can be applied to the object, including \code{AIC} (\code{\link{AIC.srafit}}), \code{logLik} (\code{\link{logLik.srafit}}), \code{vcov} (\code{\link{vcov.srafit}}), \code{coef} (\code{\link{coef.srafit}}) \code{confint} (\code{\link{confint.srafit}}), and \code{plot} (\code{\link{plot.srafit}}).
}
\references{ 
Le Rouzic, A., Houle, D., and Hansen, T.F. (2011) A modelling framework for the analysis of artificial selection-response time series. Genetics Research. 

Lande, R., and Arnold, S. (1983) The measurement of selection on correlated characters. Evolution 37:1210-1226.
}
\author{ Arnaud Le Rouzic }

\seealso{ \code{\link{sraCstvar}}, \code{\link{sraDrift}} and all other mechanistic models, \code{\link{sraAutoregTsMinuslogL}} and \code{\link{sraAutoregTimeseries}} for some details about the internal functions, \code{\link{AIC.srafit}}, \code{\link{logLik.srafit}}, \code{\link{vcov.srafit}}, \code{\link{coef.srafit}}, \code{\link{confint.srafit}}, \code{\link{plot.srafit}} for the analysis of the results. }
\examples{
# Making the example reproducible
########### Generating a dummy dataset ################

m <- c(12,11,12,14,18,17,19,22,20,19)
v <- c(53,47,97,155,150,102,65,144,179,126)
s <- c(15,14,14,17,21,20,22,25,24,NA)
n <- c(100,80,120,60,100,90,110,80,60,100)

########## Making a sra data set #######################
data <- sraData(phen.mean=m, phen.var=v, phen.sel=s, N=n)

#################### Data Analysis ####################

# Autoregressive models
autor <- sraAutoreg(data)

# Details of the model:
AIC(autor)
coef(autor)
plot(autor)
plot(autor, var=TRUE)

# Alternative scales
autor.log <- sraAutoregLog(data)
autor.herit <- sraAutoregHerit(data)
autor.evolv <- sraAutoregEvolv(data)

# Changes in the complexity of the model:
autor0 <- sraAutoreg(data, active=c(TRUE,TRUE,FALSE,FALSE))
                         
# In case of convergence issues
autor1 <- sraAutoreg(data, active=c(TRUE,TRUE,TRUE,TRUE), rep=2, rand=0.1)

}

\keyword{ models }
\keyword{ nonlinear }
\keyword{ ts }

