% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bk_cusum.R
\name{bk_cusum}
\alias{bk_cusum}
\title{Continuous time BK-CUSUM}
\usage{
bk_cusum(data, theta, coxphmod, cbaseh, ctimes, h, stoptime, C,
  twosided = FALSE, pb = FALSE, assist)
}
\arguments{
\item{data}{A \code{data.frame} with rows representing subjects and the
following named columns: \describe{
\item{\code{entrytime}:}{time of entry into study (numeric);}
\item{\code{survtime}:}{time from entry until event (numeric);}
\item{\code{censorid}:}{censoring indicator (0 = right censored, 1 = observed),
(integer).}
} and optionally additional covariates used for risk-adjustment.}

\item{theta}{The expected log-hazard ratio \eqn{\theta}{\theta} under the alternative hypothesis.
If \eqn{\theta >= 0}{\theta >= 0}, the chart will try to detect an increase
in hazard ratio (upper one-sided). If \eqn{\theta < 0}{\theta < 0},
the chart will look for a decrease in hazard ratio (lower one-sided).}

\item{coxphmod}{A Cox proportional hazards regression model as
produced by
the function \code{\link[survival:coxph]{coxph()}}. Suggested: \cr
\code{coxph(Surv(survtime, censorid) ~ covariates, data = data)}. \cr
Alternatively, a list with the following elements:
\describe{
\item{\code{formula}:}{a \code{\link[stats:formula]{formula()}} in the form \code{~ covariates};}
\item{\code{coefficients}:}{a named vector specifying risk adjustment coefficients
for covariates. Names must be the same as in \code{formula} and colnames of \code{data}.}
}}

\item{cbaseh}{A function that returns the unadjusted cumulative
baseline hazard \eqn{H_0(t)}{H_0(t)}. If \code{cbaseh} is missing but
\code{coxphmod} has been
specified as a survival object, this baseline hazard rate will be determined
using the provided \code{coxphmod}.}

\item{ctimes}{(optional): Vector of construction times at which the value of the chart should be
determined. When not specified, the chart is constructed at all failure times.}

\item{h}{(optional): Value of the control limit. The chart will only be
constructed until the value of the control limit has been reached or
surpassed.}

\item{stoptime}{(optional): Time after which the value of the chart should no
longer be determined. Default = max(failure time). Useful when ctimes
has not been specified.}

\item{C}{(optional): A numeric value indicating how long after entering the study
patients should no longer influence the value of the chart. This is
equivalent to right-censoring every observation at time \code{entrytime} + C.}

\item{twosided}{(optional): A boolean indicating whether a two-sided CUSUM
should be constructed.
If \code{TRUE}, 2 CUSUM charts are determined. One to check for an increase
of \eqn{e^\theta}{exp(\theta)} and one for
a decrease of \eqn{e^{-\theta}}{exp(-\theta)} in the hazard rate w.r.t.
the baseline hazard. Default is \code{FALSE}.}

\item{pb}{(optional): A boolean indicating whether a progress bar should
be shown. Default is \code{FALSE}.}

\item{assist}{(optional): Output of the function \code{\link[success:parameter_assist]{parameter_assist()}}}
}
\value{
An object of class \code{bkcusum} containing:
\itemize{
\item \code{BK}: a \code{data.frame} containing the following named columns:
\describe{
\item{\code{time}:}{times at which chart is constructed;}
\item{\code{value}:}{value of the chart at corresponding times.}
}
\item \code{stopind}: indicator for whether the chart was stopped by
the control limit;
\item \code{call}: the call used to obtain output;
\item \code{h}: Specified value for the control limit.
}
}
\description{
This function implements the BK-CUSUM procedure based on the
Biswas & Kalbfleisch (2008) CUSUM. To construct the Biswas & Kalbfleisch
(2008) CUSUM, set \code{C = 1} (years) or \code{C = 365} (days).
For detection purposes, it is sufficient
to determine the value of the chart at the times of failure. This can be
achieved by leaving \code{ctimes} unspecified.
The function requires the specification of \code{theta} and
has two vital parameters, at least one of which must be specified:
\itemize{
\item{\code{coxphmod}: }{Cox proportional hazards model to be used for
risk-adjustment. If \code{cbaseh} is not specified, it will be determined
from \code{coxphmod} numerically.}
\item{\code{cbaseh}: }{The cumulative baseline hazard rate to use for chart
construction. If specified with \code{coxphmod} missing, no risk-adjustment
will be performed}
}
}
\details{
The BK-CUSUM can be used to test the alternative hypothesis of an
instant change of fixed size \eqn{e^\theta}{exp(\theta)}
in the subject specific hazard rate from \eqn{h_i(t)}{h_i(t)} to
\eqn{h_i(t) e^\theta}{h_i(t) exp(\theta)}. The parameter \code{C} can be used
to ignore the contributions of subjects, C time units after their entry
into the study.
The BK-CUSUM is constructed as
\deqn{G(t) = \max_{0 \leq k \leq t} \left( \theta N(k,t) - \left( e^\theta -1  \right) \Lambda(k,t)  \right),}{G(t) = max_{0 <= k <= t} (\theta N(k,t) - (e^\theta -1) \Lambda(k,t)),}
where \eqn{\theta}{\theta} is the log expected hazard ratio,
\deqn{N(k,t) = N(t) - N(k)}{N(k,t) = N(t)-N(k)}
with \eqn{N(t)}{N(t)} the counting process of all failures at time t, and \deqn{\Lambda(k,t) = \Lambda(t) - \Lambda(k)}{\Lambda(k,t) = \Lambda(t) - \Lambda(k)}
with \eqn{\Lambda(t)}{\Lambda(t)} the summed cumulative intensity of all
subjects at time \eqn{t}{t}.
}
\examples{
require(survival)
#Select only the data of the first hospital in the surgerydat data set
tdat <- subset(surgerydat, unit == 1)

#We know that the cumulative baseline hazard in the data set is
#Exponential(0.01). If you don't know the cumulative baseline, we suggest
#leaving the cbaseh argument empty and determining a coxphmod (see help)
tcbaseh <- function(t) chaz_exp(t, lambda = 0.01)

#Determine a risk-adjustment model using a Cox proportional hazards model.
#Outcome (survival) is regressed on the available covariates:
exprfit <- as.formula("Surv(survtime, censorid) ~ age + sex + BMI")
tcoxmod <- coxph(exprfit, data= surgerydat)

#Determine the values of the chart
bk <- bk_cusum(data = tdat, theta = log(2), coxphmod = tcoxmod, cbaseh = tcbaseh, pb = TRUE)
#plot the BK-CUSUM (exact hazard)
plot(bk)

#Alternatively, cbaseh can be left empty when specifying coxphmod through coxph()
bk_cox <- bk_cusum(data = tdat, theta = log(2), coxphmod = tcoxmod, pb = TRUE)
#plot the BK-CUSUM (estimated hazard from coxph)
plot(bk_cox)
}
\references{
Biswas P. and Kalbfleisch J.D. (2008), A risk-adjusted CUSUM in
continuous time based on the Cox Model, Statistics in medicine 27, 3452-3452.
\doi{10.1002/sim.3216}
}
\seealso{
\code{\link[success]{plot.bkcusum}}, \code{\link[success]{runlength.bkcusum}}

Other quality control charts: 
\code{\link{bernoulli_cusum}()},
\code{\link{cgr_cusum}()},
\code{\link{funnel_plot}()}
}
\author{
Daniel Gomon
}
\concept{quality control charts}
