% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/proportion_transmission.R
\name{proportion_transmission}
\alias{proportion_transmission}
\title{Estimate what proportion of cases cause a certain proportion of
transmission}
\usage{
proportion_transmission(
  R,
  k,
  percent_transmission,
  method = c("p_80", "t_20"),
  simulate = FALSE,
  ...,
  offspring_dist,
  format_prop = TRUE
)
}
\arguments{
\item{R}{A \code{number} specifying the R parameter (i.e. average secondary cases
per infectious individual).}

\item{k}{A \code{number} specifying the  k parameter (i.e. overdispersion in
offspring distribution from fitted negative binomial).}

\item{percent_transmission}{A \code{number} of the percentage transmission
for which a proportion of cases has produced.}

\item{method}{A \code{character} string defining which method is used to calculate
the proportion of transmission. Options are \code{"p_80"} (default) or \code{"t_20"}.
See details for more information on each of these methods.}

\item{simulate}{A \code{logical} whether the calculation should be done
numerically (i.e. simulate secondary contacts) or analytically. Default is
\code{FALSE} which uses the analytical calculation.}

\item{...}{\link{dots} not used, extra arguments supplied will cause a warning.}

\item{offspring_dist}{An \verb{<epiparameter>} object. An S3 class for working
with epidemiological parameters/distributions, see
\code{\link[epiparameter:epiparameter]{epiparameter::epiparameter()}}.}

\item{format_prop}{A \code{logical} determining whether the proportion column
of the \verb{<data.frame>} returned by the function is formatted as a string with
a percentage sign (\verb{\%}), (\code{TRUE}, default), or as a \code{numeric} (\code{FALSE}).}
}
\value{
A \verb{<data.frame>} with the value for the proportion of cases for a
given value of R and k.
}
\description{
Calculates the proportion of cases that cause a certain
percentage of transmission.

It is commonly estimated what proportion of cases cause 80\% of transmission
(i.e. secondary cases).
This can be calculated using \code{proportion_transmission()} at varying values of
\eqn{R} and for different values of percentage transmission.

There are two methods for calculating the proportion of transmission,
\eqn{p_{80}} (default) and \eqn{t_{20}}, see \code{method} argument or details for
more information.
}
\details{
Calculates the expected proportion of transmission from a given
proportion of infectious cases. There are two methods to calculate this with
distinct formulations, \eqn{p_{80}} and \eqn{t_{20}} these can be specified
by the \code{method} argument.

\code{method = p_80} calculates relative transmission heterogeneity
from the offspring distribution of secondary cases, \eqn{Z}, where the upper
proportion of the distribution comprise \eqn{x\%} of total number of cases
given R0 and k, where \eqn{x} is typically defined as 0.8 or 80\%. e.g. 80\%
of all transmissions are generated by the upper 20\% of cases, or
\code{p_80 = 0.2}, per the 80/20 rule. In this formulation, changes in R can
have a significant effect on the estimate of \eqn{p_80} even when k is
constant. Importantly, this formulation \strong{does not} allow for true
homogeneity when \code{k = Inf} i.e. \eqn{p_{80} = 0.8}.

\code{method = t_20} calculates a similar ratio, instead in terms of
the theoretical individual reproductive number and infectiousness given
R0 and k. The individual reproductive number, 'v', is described in
Lloyd-Smith JO et al. (2005), "as a random variable representing the
expected number of secondary cases caused by a particular infected
individual. Values for v are drawn from a continuous gamma probability
distribution with population mean R0 and dispersion parameter k, which
encodes all variation in infectious histories of individuals, including
properties of the host and pathogen and environmental circumstances." The
value of k corresponds to the shape parameters of the gamma distribution
which encodes the variation in the gamma-poisson mixture aka the negative
binomial

For \code{method = t_20}, we define the upper proportion of infectiousness,
which is typically 0.2 i.e. the upper 20\% most infectious
cases, again per the 80/20 rule. e.g. the most infectious 20\% of cases,
are expected to produce 80\% of all infections, or \code{t_20 = 0.8}. Unlike
\code{method = p_80}, changes in R have no effect on the estimate
of t_80 when k is constant, but R is still required for the underlying
calculation. This formulation \strong{does} allow for true homogeneity when
k = Inf i.e. t_20 = 0.2, or t_80 = 0.8.

Multiple values of R and k can be supplied and a \verb{<data.frame>} of
every combination of these will be returned.

The numerical calculation for \code{method = p_80} uses random number generation
to simulate secondary contacts so the answers may minimally vary between
calls. The number of simulation replicates is fixed to 10\if{html}{\out{<sup>}}5\if{html}{\out{</sup>}}.
}
\examples{
# example of single values of R and k
percent_transmission <- 0.8 # 80\% of transmission
R <- 2
k <- 0.5
proportion_transmission(
  R = R,
  k = k,
  percent_transmission = percent_transmission
)

# example with multiple values of k
k <- c(0.1, 0.2, 0.3, 0.4, 0.5, 1)
proportion_transmission(
  R = R,
  k = k,
  percent_transmission = percent_transmission
)

# example with vectors of R and k
R <- c(1, 2, 3)
proportion_transmission(
  R = R,
  k = k,
  percent_transmission = percent_transmission
)
}
\references{
The analytical calculation is from:

Endo, A., Abbott, S., Kucharski, A. J., & Funk, S. (2020)
Estimating the overdispersion in COVID-19 transmission using outbreak
sizes outside China. Wellcome Open Research, 5.
\doi{10.12688/wellcomeopenres.15842.3}

The \eqn{t_{20}} method follows the formula defined in section 2.2.5 of the
supplementary material for:

Lloyd-Smith JO, Schreiber SJ, Kopp PE, Getz WM. Superspreading and the
effect of individual variation on disease emergence. Nature. 2005
Nov;438(7066):355–9. \doi{10.1038/nature04153}

The original code for the \eqn{t_{20}} method is from ongoing work
originating from \url{https://github.com/dcadam/kt} and:

Adam D, Gostic K, Tsang T, Wu P, Lim WW, Yeung A, et al.
Time-varying transmission heterogeneity of SARS and COVID-19 in Hong Kong.
2022. \doi{10.21203/rs.3.rs-1407962/v1}
}
