% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggsurvplot.R
\name{ggsurvplot}
\alias{ggsurvplot}
\title{Drawing survival curves using ggplot2}
\usage{
ggsurvplot(fit, fun = NULL, color = NULL, palette = NULL,
  break.time.by = NULL, surv.scale = c("default", "percent"),
  conf.int = FALSE, conf.int.fill = "gray", censor = TRUE, pval = FALSE,
  pval.size = 5, pval.coord = c(NULL, NULL), main = NULL, xlab = "Time",
  ylab = "Survival probability", xlim = NULL, ylim = NULL,
  legend = c("top", "bottom", "left", "right", "none"),
  legend.title = "strata", legend.labs = NULL, risk.table = FALSE,
  risk.table.col = "black", risk.table.adj = NULL,
  risk.table.height = 0.25, surv.plot.adj = NULL, surv.plot.height = 2,
  ggtheme = ggplot2::theme_classic(), ...)
}
\arguments{
\item{fit}{an object of class survfit.}

\item{fun}{an arbitrary function defining a transformation of the
survival curve. For example use function(y){y*100}. Often used transformations
can be specified with a character argument instead: "event" plots cumulative
events (f(y) = 1-y), "cumhaz" plots the cumulative hazard function (f(y) =
-log(y)), and "cloglog" creates a complimentary log-log survival plot (f(y)
= log(-log(y)) along with log scale for the x-axis).}

\item{color}{color to be used for the survival curves. This argument is
ignored when the number of strata (groups > 1). In this case, use the
argument palette.}

\item{palette}{the color palette to be used. Allowed values include "grey" for
grey color palettes; brewer palettes e.g. "RdBu", "Blues", ...; or custom
color palette e.g. c("blue", "red").}

\item{break.time.by}{numeric value controlling time axis breaks. Default value
is NULL.}

\item{surv.scale}{scale transformation of survival curves. Allowed values are
"default" or "percent".}

\item{conf.int}{logical value. If TRUE, plots confidence interval.}

\item{conf.int.fill}{fill color to be used for confidence interval.}

\item{censor}{logical value. If TRUE, censors will be drawn.}

\item{pval}{logical value. If TRUE, the p-value is added on the plot.}

\item{pval.size}{numeric value specifying the p-value text size. Default is 5.}

\item{pval.coord}{numeric vector, of length 2, specifying the x and y
coordinates of the p-value. Default values are NULL.}

\item{main, xlab, ylab}{main title and axis labels}

\item{xlim, ylim}{x and y axis limits e.g. xlim = c(0, 1000), ylim = c(0, 1).}

\item{legend}{character specifying legend position. Allowed values are one of
c("top", "bottom", "left", "right", "none"). Default is "top" side
position. to remove the legend use legend = "none". Legend position can be
also specified using a numeric vector c(x, y); see details section.}

\item{legend.title}{legend title.}

\item{legend.labs}{character vector specifying legend labels.}

\item{risk.table}{logical value specifying whether to show risk table. Default
is FALSE.}

\item{risk.table.col}{color to be used for risk table. Default value is
"black". If you want to color by strata (i.e. groups), use risk.table.col =
"strata".}

\item{risk.table.adj}{numeric value, used to adjust the location of the risk
table. Negative value will shift the table to the left and positive value
will shift the table to the right side. Ignored when risk.table = FALSE.}

\item{risk.table.height}{the height of the risk table on the grid. Increase
the value when you have many strata. Default is 0.25. Ignored when
risk.table = FALSE.}

\item{surv.plot.adj}{numeric value, used to adjust survival plot (like
risk.table.adj). Ignored when risk.table = FALSE.}

\item{surv.plot.height}{the height of the survival plot on the grid. Default
is 2. Ignored when risk.table = FALSE.}

\item{ggtheme}{function, ggplot2 theme name. Default value is theme_classic().
Allowed values include ggplot2 official themes: theme_gray(), theme_bw(),
theme_minimal(), theme_classic(), theme_void(), ....}

\item{...}{other arguments to be passed to ggplot2 geom_*() functions such as
linetype, size, ...}
}
\value{
return a ggplot2 (when risk.table = FALSE).
}
\description{
Drawing survival curves using ggplot2
}
\details{
\strong{legend position}: The argument \strong{legend} can be also a
 numeric vector c(x,y). In this case it is possible to position the legend
 inside the plotting area. x and y are the coordinates of the legend box.
 Their values should be between 0 and 1. c(0,0) corresponds to the "bottom
 left" and c(1,1) corresponds to the "top right" position. For instance use
 legend = c(0.8, 0.2).
}
\examples{

#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%
# Example 1: Survival curves with two groups
#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%

# Fit survival curves
#++++++++++++++++++++++++++++++++++++
require("survival")
fit<- survfit(Surv(time, status) ~ sex, data = lung)

# Drawing survival curves
ggsurvplot(fit)

# Legend: title, labels and position
#++++++++++++++++++++++++++++++++++++

# Change the legend title and labels
ggsurvplot(fit, legend = "bottom",
          legend.title = "Sex",
          legend.labs = c("Male", "Female"))

# Specify legend position by its coordinates
ggsurvplot(fit, legend = c(0.2, 0.2))


# format
#++++++++++++++++++++++++++++++++++++
# change line size --> 1
# Change line types by groups (i.e. "strata")
# and change color palette
ggsurvplot(fit,  size = 1,  # change line size
          linetype = "strata", # change line type by groups
          break.time.by = 250, # break time axis by 250
          palette = c("#E7B800", "#2E9FDF"), # custom color palette
          conf.int = TRUE, # Add confidence interval
          pval = TRUE # Add p-value
)

# Use brewer color palette "Dark2"
ggsurvplot(fit, linetype = "strata",
          conf.int = TRUE, pval = TRUE,
          palette = "Dark2")


# Add risk table
#++++++++++++++++++++++++++++++++++++

# Add Risk table
ggsurvplot(fit, pval = TRUE, conf.int = TRUE,
          risk.table = TRUE)

# Change color, linetype by strata, risk.table color by strata
ggsurvplot(fit,
          pval = TRUE, conf.int = TRUE,
          risk.table = TRUE, # Add risk table
          risk.table.col = "strata", # Change risk table color by groups
          lienetype = "strata", # Change line type by groups
          ggtheme = theme_bw(), # Change ggplot2 theme
          palette = c("#E7B800", "#2E9FDF"))

# Survival curve transformation
#++++++++++++++++++++++++++++++++++++
# Plot cumulative events
ggsurvplot(fit, conf.int = TRUE,
          palette = c("#FF9E29", "#86AA00"),
          risk.table = TRUE, risk.table.col = "strata",
          fun = "event")


# Arbitrary function
ggsurvplot(fit, conf.int = TRUE,
          palette = c("#FF9E29", "#86AA00"),
          risk.table = TRUE, risk.table.col = "strata",
          pval = TRUE,
          fun = function(y) y*100)


#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%
# Example 3: Survival curve with multiple group
#\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%\%

# Fit (complexe) survival curves
#++++++++++++++++++++++++++++++++++++

require("survival")
fit2 <- survfit( Surv(time, status) ~ rx + adhere,
                data = colon )

# Visualize
#++++++++++++++++++++++++++++++++++++

# Visualize: add p-value, chang y limits
# change color using brewer palette
ggsurvplot(fit2, pval = TRUE,
          break.time.by = 400,
          risk.table = TRUE)

# Adjust risk table and survival plot locations
# ++++++++++++++++++++++++++++++++++++
# Adjust risk table location, shift to the left

ggsurvplot(fit2, pval = TRUE,
          break.time.by = 400,
          risk.table = TRUE,
          risk.table.col = "strata",
          risk.table.adj = -2, # risk table location adj
          palette = "Dark2")

# Adjust survival plot location, shift to the right
# Change Risk table height
ggsurvplot(fit2, pval = TRUE,
          break.time.by = 400,
          risk.table = TRUE,
          risk.table.col = "strata",
          risk.table.height = 0.5, # Useful when you have multiple groups
          surv.plot.adj = 4.9, # surv plot location adj
          palette = "Dark2")


# Change legend labels
# ++++++++++++++++++++++++++++++++++++
\dontrun{
ggsurvplot(fit2, pval = TRUE,
          break.time.by = 400,
          risk.table = TRUE,
          risk.table.col = "strata",
          ggtheme = theme_bw(),
          legend.labs = c("A", "B", "C", "D", "E", "F"))
}

}

