% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/generalized_bootstrap.R
\name{rescale_reps}
\alias{rescale_reps}
\title{Rescale replicate factors to have specified lower bound}
\usage{
rescale_reps(x, tau = "auto", min_wgt = 0.01, digits = 2)
}
\arguments{
\item{x}{Either a replicate survey design object,
or a numeric matrix of replicate weights.}

\item{tau}{Either \code{"auto"}, or a single number. This is the rescaling constant
used to avoid negative weights through the transformation \eqn{\frac{w + \tau - 1}{\tau}},
where \eqn{w} is the original weight and \eqn{\tau} is the rescaling constant \code{tau}. \cr
If \code{tau="auto"}, the rescaling factor is determined automatically as follows:
if all of the adjustment factors exceed the minimum value \code{min_wgt},
 then \code{tau} is set equal to 1;
otherwise, \code{tau} is set to the smallest value needed to rescale
the adjustment factors such that they are all at least \code{min_wgt}.}

\item{min_wgt}{Only used if \code{tau='auto'}. Specifies the minimum acceptable value for the rescaled weights.
Must be at least zero and must be less than one.}

\item{digits}{Only used if \code{tau='auto'}. Specifies the number of decimal places
to use for choosing \code{tau}. Using a smaller number of \code{digits}
is useful simply for producing easier-to-read documentation.}
}
\value{
If the input is a numeric matrix, returns the rescaled matrix.
If the input is a replicate survey design object, returns an updated replicate survey design object.

For a replicate survey design object, results depend on
whether the object has a matrix of replicate factors rather than
a matrix of replicate weights (which are the product of replicate factors and sampling weights).
If the design object has \code{combined.weights=FALSE},
then the replication factors are adjusted.
If the design object has \code{combined.weights=TRUE},
then the replicate weights are adjusted. It is strongly
recommended to only use the rescaling method for replication factors
rather than the weights.

For a replicate survey design object, the \code{scale} element
of the design object will be updated appropriately,
and an element \code{tau} will also be added.
If the input is a matrix instead of a survey design object,
the result matrix will have an attribute named \code{tau}
which can be retrieved using \code{attr(x, 'tau')}.
}
\description{
Rescale replicate factors to ensure that they all exceed
a specified lower bound. The main use of this rescaling is to ensure
that all replicate weights are strictly positive.

Note that this rescaling has no impact on variance estimates for totals,
but variance estimates for nonlinear statistics will be affected by the rescaling.
}
\details{
Let \eqn{\mathbf{A} = \left[ \mathbf{a}^{(1)} \cdots \mathbf{a}^{(b)} \cdots \mathbf{a}^{(B)} \right]} denote the \eqn{(n \times B)} matrix of replicate adjustment factors.
To eliminate negative adjustment factors, Beaumont and Patak (2012) propose forming a rescaled matrix of nonnegative replicate factors \eqn{\mathbf{A}^S} by rescaling each adjustment factor \eqn{a_k^{(b)}} as follows:
\deqn{
   a_k^{S,(b)} = \frac{a_k^{(b)} + \tau - 1}{\tau}
 }
where \eqn{\tau \geq 1 - a_k^{(b)} \geq 1} for all \eqn{k} in \eqn{\left\{ 1,\ldots,n \right\}} and all \eqn{b} in \eqn{\left\{1, \ldots, B\right\}}.

The value of \eqn{\tau} can be set based on the realized adjustment factor matrix \eqn{\mathbf{A}} or by choosing \eqn{\tau} prior to generating the adjustment factor matrix \eqn{\mathbf{A}} so that \eqn{\tau} is likely to be large enough to prevent negative adjustment factors.

If the adjustment factors are rescaled in this manner, it is important to adjust the scale factor used in estimating the variance with the bootstrap replicates.
For example, for bootstrap replicates, the adjustment factor becomes \eqn{\frac{\tau^2}{B}} instead of \eqn{\frac{1}{B}}.
\deqn{
 \textbf{Prior to rescaling: } v_B\left(\hat{T}_y\right) = \frac{1}{B}\sum_{b=1}^B\left(\hat{T}_y^{*(b)}-\hat{T}_y\right)^2
 }
\deqn{
 \textbf{After rescaling: } v_B\left(\hat{T}_y\right) = \frac{\tau^2}{B}\sum_{b=1}^B\left(\hat{T}_y^{S*(b)}-\hat{T}_y\right)^2
}
}
\examples{
# Example 1: Rescaling a matrix of replicate weights

 rep_wgts <- matrix(
   c(1.69742746694909, -0.230761178913411, 1.53333377634192,
     0.0495043413294782, 1.81820367441039, 1.13229198793703,
     1.62482013925955, 1.0866133494029, 0.28856654131668,
     0.581930729719006, 0.91827012312825, 1.49979905894482,
     1.26281337410693, 1.99327362761477, -0.25608700039304),
   nrow = 3, ncol = 5
 )

 rescaled_wgts <- rescale_reps(rep_wgts, tau = 'auto', min_wgt = 0.01)

 print(rep_wgts)
 print(rescaled_wgts)

 # Example 2: Rescaling replicate weights of a survey design object
 set.seed(2023)
 library(survey)
 data('mu284', package = 'survey')

 ## First create a bootstrap design object
 svy_design_object <- svydesign(
   data = mu284,
   ids = ~ id1 + id2,
   fpc = ~ n1 + n2
 )

 boot_design <- as_gen_boot_design(
   design = svy_design_object,
   variance_estimator = "Stratified Multistage SRS",
   replicates = 5, tau = 1
 )

 ## Rescale the weights
 rescaled_boot_design <- boot_design |>
   rescale_reps(tau = 'auto', min_wgt = 0.01)

 boot_wgts <- weights(boot_design, "analysis")
 rescaled_boot_wgts <- weights(rescaled_boot_design, 'analysis')

 print(boot_wgts)
 print(rescaled_boot_wgts)
}
\references{
This method was suggested by Fay (1989) for the specific application
of creating replicate factors using his generalized replication method.
Beaumont and Patak (2012) provided an extended discussion on this rescaling
method in the context of creating generalized bootstrap weights.
The notation used in this documentation is taken from Beaumont and Patak (2012).

- Beaumont, Jean-François, and Zdenek Patak. 2012.
"On the Generalized Bootstrap for Sample Surveys with Special Attention to Poisson Sampling: Generalized Bootstrap for Sample Surveys."
International Statistical Review 80 (1): 127–48.
https://doi.org/10.1111/j.1751-5823.2011.00166.x.
\cr \cr
- Fay, Robert. 1989. "Theory And Application Of Replicate Weighting For Variance Calculations."
In, 495–500. Alexandria, VA: American Statistical Association.
http://www.asasrms.org/Proceedings/papers/1989_033.pdf
}
