

#' col2hex
#'
#' Converts named colors to hex (html) codes.
#'
#' @param color One or more named colors. Also accepts hex colors (useful for
#'   adding alpha values).
#' @param alpha Alpha level ranging form 1.0 (opaque) to 0.0 (transparent). The
#'   default (NULL) omits alpha columns from the returned hexadecimal
#'   characters.
#' @return Hexadecimal color codes
#' @export
col2hex <- function(color, alpha = NULL) UseMethod("col2hex")


#' @export
#' @importFrom grDevices col2rgb
col2hex.character <- function(color, alpha = NULL) {
  color <- grDevices::col2rgb(color)
  col2hex(color, alpha)
}

#' @export
col2hex.numeric <- function(color, alpha = NULL) {
  stopifnot(length(color) == 3L)
  color <- col2hex_dbl(color, alpha)
  col2hex(color, alpha)
}

col2hex_dbl <- function(color, alpha = NULL) {
  color <- c(color, alpha)
  stopifnot(length(color) %in% c(3, 4))
  if (any(color > 1.0) || (sum(color[1:3] %% 1) > 0)) {
    maxColorValue <- 255
  } else {
    maxColorValue <- 1
  }
  if (length(color) == 4) {
    color[4] <- color[4] * maxColorValue
  } else {
    color[4] <- 1
  }
  grDevices::rgb(
    color[1], color[2], color[3], color[4], maxColorValue = maxColorValue)
}


#' @export
col2hex.integer <- function(color, alpha = NULL) {
  color <- as.numeric(color)
  col2hex(color, alpha)
}

#' @export
col2hex.matrix <- function(color, alpha = NULL) {
  stopifnot(nrow(color) == 3L)
  if (all(c("h", "s", "v") %in% row.names(color))) {
    coltype <- "hsv"
  } else {
    coltype <- "rgb"
  }
  matrix2col(coltype, color, alpha)
}


#' gg_cols
#'
#' @description Generates colors like those generated by
#'  '\pkg{ggplot2}.
#' @param n Number of colors (HEX values) to return.
#' @return Character vector of HEX color codes.
#'
#' @examples
#'
#' gg_cols(8)
#' @keywords internal
#' @importFrom grDevices hcl
#' @export
gg_cols <- function(n) {
  hues = seq(15, 375, length = n + 1)
  hcl(h = hues, l = 65, c = 100)[1:n]
}


#' @importFrom grDevices rgb col2rgb
alphacolor <- function(cols, a = .99) {
  cols <- t(col2rgb(cols, alpha = TRUE)) / 255
  rgb(cols, alpha = a)
}

#' @importFrom grDevices col2rgb
is.color <- function(x) {
  if (all(grepl("^#", x))) return(TRUE)
  x <- tryCatch(col2rgb(x),
                error = function(e) return(NULL))
  if (!is.null(x)) return(TRUE)
  FALSE
}

matrix2col <- function(f, x, alpha = NULL) {
  apply(x, 2, function(i) {
    args <- as.list(i)
    if (is.integer(i)) {
      args$maxColorValue <- 255
    } else {
      args$maxColorValue <- 1L
    }
    if (!is.null(alpha)) {
      args$alpha <- alpha * args$maxColorValue
    }
    do.call(f, as.list(args))
  })
}

colors_and_matrices <- function(color, alpha = NULL) {
  if (all(c("h", "s", "v") %in% row.names(color))) {
    color <- matrix2col("hsv", color, alpha)
  } else if (all(c("red", "green", "blue") %in% row.names(color))) {
    color <- matrix2col("rgb", color, alpha)
  }
  color
}
