% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/helpers.R
\name{erase_water}
\alias{erase_water}
\title{Erase water area from an input polygon dataset}
\usage{
erase_water(input_sf, area_threshold = 0.75, year = NULL)
}
\arguments{
\item{input_sf}{An input sf object, ideally obtained with the tigris package or through tidycensus.}

\item{area_threshold}{The percentile rank cutoff of water areas to use in the erase operation, ranked by size. Defaults to 0.75, representing the water areas in the 75th percentile and up (the largest 25 percent of areas).  This value may need to be modified by the user to achieve optimal results for a given location.}

\item{year}{The year to use for the water layer; defaults to 2020 unless the \code{tigris_year} option is otherwise set.}
}
\value{
An output sf object representing the polygons in \code{input_sf} with water areas erased.
}
\description{
This function 'erases' water area from an input polygon dataset (typically a Census dataset).
'Erase' is defined in the traditional GIS sense as the removal of areas in an input layer
from an erase layer, returning the modified input layer. A common use-case is to improve
cartographic representation of locations where US Census polygons include more water area
than desired (e.g. New York City, Seattle) or to support contiguity-based spatial analyses that
might otherwise incorrectly assume that polygons across bodies of water are neighbors.
}
\details{
The function works by identifying US counties that intersect the input polygon layer,
then requesting water polygons (using \code{tigris::area_water()}) to be erased from
those input polygons. The \code{area_threshold} parameter can be tuned to determine the
percentile ranking of bodies of water (by area) to use;
the default is a percentile ranking of 0.75, erasing the largest 25 percent of water
bodies in the region.

Analysts will ideally have transformed the input coordinate reference system (CRS) of their data
to a projected CRS to improve performance; see \url{https://walker-data.com/census-r/census-geographic-data-and-applications-in-r.html#coordinate-reference-systems} for more information on
how to perform CRS transformations.  Analysts should also use this function with caution;
the function may generate sliver polygons or irregular geometries in the output layer,
especially if the input sf object was not obtained with the tigris package.  Also, the operation
may be quite slow for large input areas.
}
\examples{
\dontrun{

library(tigris)
library(sf)
options(tigris_use_cache = TRUE)

king_tracts <- tracts(state = "WA", county = "King", year = 2020)

# CRS: NAD 1983 / Washington North (State Plane)
king_erased <- king_tracts \%>\%
  st_transform(32148) \%>\%
  erase_water(area_threshold = 0.9)

plot(king_erased$geometry)

}
}
