% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/coersion-tk_xts.R
\name{tk_xts}
\alias{tk_xts}
\alias{tk_xts_}
\title{Coerce time series objects and tibbles with date/date-time columns to xts.}
\usage{
tk_xts(data, select = NULL, date_var = NULL, silent = FALSE, ...)

tk_xts_(data, select = NULL, date_var = NULL, silent = FALSE, ...)
}
\arguments{
\item{data}{A time-based tibble or time-series object.}

\item{select}{\strong{Applicable to tibbles and data frames only}.
The column or set of columns to be coerced to \code{ts} class.}

\item{date_var}{\strong{Applicable to tibbles and data frames only}.
Column name to be used to \code{order.by}.
\code{NULL} by default. If \code{NULL}, function will find the date or date-time column.}

\item{silent}{Used to toggle printing of messages and warnings.}

\item{...}{Additional parameters to be passed to \code{xts::xts()}. Refer to \code{xts::xts()}.}
}
\value{
Returns a \code{xts} object.
}
\description{
Coerce time series objects and tibbles with date/date-time columns to xts.
}
\details{
\code{tk_xts} is a wrapper for \code{xts::xts()} that is designed
to coerce \code{tibble} objects that have a "time-base" (meaning the values vary with time)
to \code{xts} class objects. There are three main advantages:
\enumerate{
\item Non-numeric columns that are not removed via \code{select} are dropped and the user is warned.
This prevents an error or coercion issue from occurring.
\item The date column is auto-detected if not specified by \code{date_var}. This takes
the effort off the user to assign a date vector during coercion.
\item \code{ts} objects are automatically coerced if a "timetk index" is present. Refer to \code{\link[=tk_ts]{tk_ts()}}.
}

The \code{select} argument can be used to select subsets
of columns from the incoming data.frame.
Only columns containing numeric data are coerced.
The \code{date_var} can be used to specify the column with the date index.
If \code{date_var = NULL}, the date / date-time column is interpreted.
Optionally, the \code{order.by} argument from the underlying \code{xts::xts()} function can be used.
The user must pass a vector of dates or date-times if \code{order.by} is used.

For non-data.frame object classes (e.g. \code{xts}, \code{zoo}, \code{timeSeries}, etc) the objects are coerced
using \code{xts::xts()}.

\code{tk_xts_} is a nonstandard evaluation method.
}
\examples{
library(tidyverse)
library(timetk)

### tibble to xts: Comparison between tk_xts() and xts::xts()
data_tbl <- tibble::tibble(
    date = seq.Date(as.Date("2016-01-01"), by = 1, length.out = 5),
    x    = rep("chr values", 5),
    y    = cumsum(1:5),
    z    = cumsum(11:15) * rnorm(1))

# xts: Character columns cause coercion issues; order.by must be passed a vector of dates
xts::xts(data_tbl[,-1], order.by = data_tbl$date)

# tk_xts: Non-numeric columns automatically dropped; No need to specify date column
tk_xts(data_tbl)

# ts can be coerced back to xts
data_tbl \%>\%
    tk_ts(start = 2016, freq = 365) \%>\%
    tk_xts()

### Using select and date_var
tk_xts(data_tbl, select = y, date_var = date)


### NSE: Enables programming
date_var <- "date"
select   <- "y"
tk_xts_(data_tbl, select = select, date_var = date_var)

}
\seealso{
\code{\link[=tk_tbl]{tk_tbl()}}, \code{\link[=tk_zoo]{tk_zoo()}}, \code{\link[=tk_zooreg]{tk_zooreg()}}, \code{\link[=tk_ts]{tk_ts()}}
}
