% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/linalg.R
\name{linalg_lstsq}
\alias{linalg_lstsq}
\title{Computes a solution to the least squares problem of a system of linear equations.}
\usage{
linalg_lstsq(A, B, rcond = NULL, ..., driver = NULL)
}
\arguments{
\item{A}{(Tensor): lhs tensor of shape \verb{(*, m, n)} where \code{*} is zero or more batch dimensions.}

\item{B}{(Tensor): rhs tensor of shape \verb{(*, m, k)} where \code{*} is zero or more batch dimensions.}

\item{rcond}{(float, optional): used to determine the effective rank of \code{A}.
If \code{rcond = NULL}, \code{rcond} is set to the machine
precision of the dtype of \code{A} times \code{max(m, n)}. Default: \code{NULL}.}

\item{...}{currently unused.}

\item{driver}{(str, optional): name of the LAPACK/MAGMA method to be used.
If \code{NULL}, \code{'gelsy'} is used for CPU inputs and \code{'gels'} for CUDA inputs.
Default: \code{NULL}.}
}
\value{
A list \verb{(solution, residuals, rank, singular_values)}.
}
\description{
Letting \teqn{\mathbb{K}} be \teqn{\mathbb{R}} or \teqn{\mathbb{C}},
the \strong{least squares problem} for a linear system \teqn{AX = B} with
\teqn{A \in \mathbb{K}^{m \times n}, B \in \mathbb{K}^{m \times k}} is defined as
}
\details{
\Sexpr[results=rd, stage=build]{torch:::math_to_rd("
\\\\min_{X \\\\in \\\\mathbb{K}^{n \\\\times k}} \\\\|AX - B\\\\|_F
")}

where \teqn{\|-\|_F} denotes the Frobenius norm.
Supports inputs of float, double, cfloat and cdouble dtypes.

Also supports batches of matrices, and if the inputs are batches of matrices then
the output has the same batch dimensions.
\code{driver} chooses the LAPACK/MAGMA function that will be used.

For CPU inputs the valid values are \code{'gels'}, \code{'gelsy'}, \verb{'gelsd}, \code{'gelss'}.
For CUDA input, the only valid driver is \code{'gels'}, which assumes that \code{A} is full-rank.

To choose the best driver on CPU consider:
\itemize{
\item If \code{A} is well-conditioned (its \href{https://pytorch.org/docs/master/linalg.html#torch.linalg.cond}{condition number} is not too large), or you do not mind some precision loss.
\item For a general matrix: \code{'gelsy'} (QR with pivoting) (default)
\item If \code{A} is full-rank: \code{'gels'} (QR)
\item If \code{A} is not well-conditioned.
\item \code{'gelsd'} (tridiagonal reduction and SVD)
\item But if you run into memory issues: \code{'gelss'} (full SVD).
}

See also the \href{https://netlib.org/lapack/lug/node27.html}{full description of these drivers}

\code{rcond} is used to determine the effective rank of the matrices in \code{A}
when \code{driver} is one of (\code{'gelsy'}, \code{'gelsd'}, \code{'gelss'}).
In this case, if \teqn{\sigma_i} are the singular values of \code{A} in decreasing order,
\teqn{\sigma_i} will be rounded down to zero if \teqn{\sigma_i \leq rcond \cdot \sigma_1}.
If \code{rcond = NULL} (default), \code{rcond} is set to the machine precision of the dtype of \code{A}.

This function returns the solution to the problem and some extra information in a list of
four tensors \verb{(solution, residuals, rank, singular_values)}. For inputs \code{A}, \code{B}
of shape \verb{(*, m, n)}, \verb{(*, m, k)} respectively, it cointains
\itemize{
\item \code{solution}: the least squares solution. It has shape \verb{(*, n, k)}.
\item \code{residuals}: the squared residuals of the solutions, that is, \teqn{\|AX - B\|_F^2}.
It has shape equal to the batch dimensions of \code{A}.
It is computed when \code{m > n} and every matrix in \code{A} is full-rank,
otherwise, it is an empty tensor.
If \code{A} is a batch of matrices and any matrix in the batch is not full rank,
then an empty tensor is returned. This behavior may change in a future PyTorch release.
\item \code{rank}: tensor of ranks of the matrices in \code{A}.
It has shape equal to the batch dimensions of \code{A}.
It is computed when \code{driver} is one of (\code{'gelsy'}, \code{'gelsd'}, \code{'gelss'}),
otherwise it is an empty tensor.
\item \code{singular_values}: tensor of singular values of the matrices in \code{A}.
It has shape \verb{(*, min(m, n))}.
It is computed when \code{driver} is one of (\code{'gelsd'}, \code{'gelss'}),
otherwise it is an empty tensor.
}
}
\note{
This function computes \code{X = A$pinverse() \%*\% B} in a faster and
more numerically stable way than performing the computations separately.
}
\section{Warning}{

The default value of \code{rcond} may change in a future PyTorch release.
It is therefore recommended to use a fixed value to avoid potential
breaking changes.
}

\examples{
if (torch_is_installed()) {
A <- torch_tensor(rbind(c(10, 2, 3), c(3, 10, 5), c(5, 6, 12)))$unsqueeze(1) # shape (1, 3, 3)
B <- torch_stack(list(
  rbind(c(2, 5, 1), c(3, 2, 1), c(5, 1, 9)),
  rbind(c(4, 2, 9), c(2, 0, 3), c(2, 5, 3))
), dim = 1) # shape (2, 3, 3)
X <- linalg_lstsq(A, B)$solution # A is broadcasted to shape (2, 3, 3)
}
}
\seealso{
Other linalg: 
\code{\link{linalg_cholesky_ex}()},
\code{\link{linalg_cholesky}()},
\code{\link{linalg_det}()},
\code{\link{linalg_eigh}()},
\code{\link{linalg_eigvalsh}()},
\code{\link{linalg_eigvals}()},
\code{\link{linalg_eig}()},
\code{\link{linalg_householder_product}()},
\code{\link{linalg_inv_ex}()},
\code{\link{linalg_inv}()},
\code{\link{linalg_matrix_norm}()},
\code{\link{linalg_matrix_power}()},
\code{\link{linalg_matrix_rank}()},
\code{\link{linalg_multi_dot}()},
\code{\link{linalg_norm}()},
\code{\link{linalg_pinv}()},
\code{\link{linalg_qr}()},
\code{\link{linalg_slogdet}()},
\code{\link{linalg_solve}()},
\code{\link{linalg_svdvals}()},
\code{\link{linalg_svd}()},
\code{\link{linalg_tensorinv}()},
\code{\link{linalg_tensorsolve}()},
\code{\link{linalg_vector_norm}()}
}
\concept{linalg}
