% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tsmp.R
\name{tsmp}
\alias{tsmp}
\title{Computes the Matrix Profile and Profile Index}
\usage{
tsmp(..., window_size, exclusion_zone = 1/2, mode = c("stomp", "stamp",
  "simple", "mstomp", "scrimp"), verbose = 2, s_size = Inf,
  must_dim = NULL, exc_dim = NULL, n_workers = 1,
  .keep_data = TRUE)
}
\arguments{
\item{...}{a \code{matrix} or a \code{vector}. If a second time series is supplied it will be a join matrix
profile (except for \code{\link[=mstomp]{mstomp()}}).}

\item{window_size}{an \code{int} with the size of the sliding window.}

\item{exclusion_zone}{a \code{numeric}. Size of the exclusion zone, based on window size (default is
\code{1/2}). See details.}

\item{mode}{the algorithm that will be used to compute the matrix profile. (Default is \code{stomp}).
See details.}

\item{verbose}{an \code{int}. (Default is \code{2}). See details.}

\item{s_size}{a \code{numeric}. for anytime algorithm, represents the size (in observations) the
random calculation will occur (default is \code{Inf}). See details.}

\item{must_dim}{an \code{int} or \code{vector} of which dimensions to forcibly include (default is \code{NULL}).
See details.}

\item{exc_dim}{an \code{int} or \code{vector} of which dimensions to exclude (default is \code{NULL}). See
details.}

\item{n_workers}{an \code{int}. Number of workers for parallel. (Default is \code{1}).}

\item{.keep_data}{a \code{logical}. (Default is \code{TRUE}). Keeps the data embedded to resultant object.}
}
\value{
Returns the matrix profile \code{mp} and profile index \code{pi}. It also returns the left and
right matrix profile \code{lmp}, \code{rmp} and profile index \code{lpi}, \code{rpi} that may be used to detect
Time Series Chains. \code{\link[=mstomp]{mstomp()}} returns a multidimensional Matrix Profile.
}
\description{
This is a wrap function that makes easy to use all available algorithms to compute the Matrix
Profile and Profile Index for multiple purposes.
}
\details{
The Matrix Profile, has the potential to revolutionize time series data mining because
of its generality, versatility, simplicity and scalability. In particular it has implications
for time series motif discovery, time series joins, shapelet discovery (classification),
density estimation, semantic segmentation, visualization, rule discovery, clustering etc.

The first algorithm invented was the \code{\link[=stamp]{stamp()}} that using \code{\link[=mass]{mass()}} as an ultra-fast Algorithm
for Similarity Search allowed to compute the Matrix Profile in reasonable time. One of its main
feature was its Anytime property which using a randomized approach could return a "best-so-far"
matrix that could give us the correct answer (using for example 1/10 of all iterations) almost
every time.

The next algorithm was \code{\link[=stomp]{stomp()}} that currently is the most used. Researchers noticed that the
dot products do not need to be recalculated from scratch for each subsequence. Instead, we can
reuse the values calculated for the first subsequence to make a faster calculation in the next
iterations. The idea is to make use of the intersections between the required products in
consecutive iterations. This approach reduced the time to compute the Matrix Profile to about
3% compared to \code{\link[=stamp]{stamp()}}, but on the other hand, we lost the Anytime property.

Currently there is a new algorithm that I'll not explain further here. It is called \code{\link[=scrimp]{scrimp()}},
and is as fast as \code{\link[=stomp]{stomp()}}, and have the Anytime property. This algorithm is implemented in
this package, but stills experimental, so expect minor discrepancies compared to previous
algorithms.

Further, there is the \code{\link[=mstomp]{mstomp()}} that computes a multidimensional Matrix Profile that allows to
meaningful MOTIF discovery in Multivariate Time Series. And \code{\link[=simple_fast]{simple_fast()}} that also handles
Multivariate Time Series, but focused in Music Analysis and Exploration.

Some parameters are global across the algorithms:
\describe{
\item{...}{One or two time series (except for \code{\link[=mstomp]{mstomp()}}). The second time series can be smaller than the first.}
\item{window_size}{The sliding window.}
\item{exclusion_zone}{Is used to avoid trivial matches; if a query data is provided
(join similarity), this parameter is ignored.}
\item{verbose}{Changes how much information is printed by this function; \code{0} means nothing,
\code{1} means text, \code{2} means text and sound.}
\item{n_workers}{number of threads for parallel computing (except \code{simple_fast} and \code{scrimp}).
If the value is 2 or more, the '_par' version of the algorithm will be used.}
}

\code{s_size} is used only in Anytime algorithms: \code{\link[=stamp]{stamp()}} and \code{\link[=scrimp]{scrimp()}}.
\code{must_dim} and \code{exc_dim} are used only in \code{\link[=mstomp]{mstomp()}}.
\code{mode} can be any of the following: \code{stomp}, \code{stamp}, \code{simple}, \code{mstomp}, \code{scrimp}.
}
\examples{
# default with [stomp()]
mp <- tsmp(mp_toy_data$data[1:200, 1], window_size = 30, verbose = 0)

# parallel with [stomp_par()]
mp <- tsmp(mp_toy_data$data[1:200, 1], window_size = 30, n_workers = 2, verbose = 0)

# Anytime STAMP
mp <- tsmp(mp_toy_data$data[1:200, 1], window_size = 30, mode = "stamp", s_size = 50, verbose = 0)

# [mstomp()]
mp <- tsmp(mp_toy_data$data[1:200, ], window_size = 30, mode = "mstomp", verbose = 0)

# [simple_fast()]
mp <- tsmp(mp_toy_data$data[1:200, ], window_size = 30, mode = "simple", verbose = 0)
}
\references{
\itemize{
\item Silva D, Yeh C, Batista G, Keogh E. Simple: Assessing Music Similarity Using
Subsequences Joins. Proc 17th ISMIR Conf. 2016;23–30.
}

\itemize{
\item Silva DF, Yeh C-CM, Zhu Y, Batista G, Keogh E. Fast Similarity Matrix Profile for
Music Analysis and Exploration. IEEE Trans Multimed. 2018;14(8):1–1.
}

\itemize{
\item Yeh CM, Kavantzas N, Keogh E. Matrix Profile VI : Meaningful Multidimensional Motif
Discovery.
}

\itemize{
\item Yeh CCM, Zhu Y, Ulanova L, Begum N, Ding Y, Dau HA, et al. Matrix profile I: All
pairs similarity joins for time series: A unifying view that includes motifs, discords and
shapelets. Proc - IEEE Int Conf Data Mining, ICDM. 2017;1317–22.
}

\itemize{
\item Zhu Y, Imamura M, Nikovski D, Keogh E. Matrix Profile VII: Time Series Chains: A
New Primitive for Time Series Data Mining. Knowl Inf Syst. 2018 Jun 2;1–27.
}

\itemize{
\item Zhu Y, Zimmerman Z, Senobari NS, Yeh CM, Funning G. Matrix Profile II : Exploiting
a Novel Algorithm and GPUs to Break the One Hundred Million Barrier for Time Series Motifs and
Joins. Icdm. 2016 Jan 22;54(1):739–48.
}

Website: \url{https://sites.google.com/view/simple-fast}

Website: \url{https://sites.google.com/site/ismir2016simple/home}

Website: \url{http://www.cs.ucr.edu/~eamonn/MatrixProfile.html}
}
\seealso{
Other matrix profile computations: \code{\link{mstomp_par}},
  \code{\link{scrimp}}, \code{\link{stamp_par}},
  \code{\link{stomp_par}}
}
\concept{matrix profile computations}
