#' Load conversion table for Vietnamese characters
#' @param version Version of this table (1 or 2) - 2 is currentlty being used
#' @return data.frame
#' @details
#' The table was adapted from \url{http://vietunicode.sourceforge.net/charset/}
#' @export
#'

loadEncodingTableVN <- function(version)
  {

  if(version == 1){
  out <- structure(list(Unicode_Hex = c("U+00C0",
                                        "U+00C1", "U+00C2", "U+00C3", "U+00C8", "U+00C9", "U+00CA", "U+00CC",
                                        "U+00CD", "U+00D2", "U+00D3", "U+00D4", "U+00D5", "U+00D9", "U+00DA",
                                        "U+00DD", "U+00E0", "U+00E1", "U+00E2", "U+00E3", "U+00E8", "U+00E9",
                                        "U+00EA", "U+00EC", "U+00ED", "U+00F2", "U+00F3", "U+00F4", "U+00F5",
                                        "U+00F9", "U+00FA", "U+00FD", "U+0102", "U+0103", "U+0110", "U+0111",
                                        "U+0128", "U+0129", "U+0168", "U+0169", "U+01A0", "U+01A1", "U+01AF",
                                        "U+01B0", "U+1EA0", "U+1EA1", "U+1EA2", "U+1EA3", "U+1EA4", "U+1EA5",
                                        "U+1EA6", "U+1EA7", "U+1EA8", "U+1EA9", "U+1EAA", "U+1EAB", "U+1EAC",
                                        "U+1EAD", "U+1EAE", "U+1EAF", "U+1EB0", "U+1EB1", "U+1EB2", "U+1EB3",
                                        "U+1EB4", "U+1EB5", "U+1EB6", "U+1EB7", "U+1EB8", "U+1EB9", "U+1EBA",
                                        "U+1EBB", "U+1EBC", "U+1EBD", "U+1EBE", "U+1EBF", "U+1EC0", "U+1EC1",
                                        "U+1EC2", "U+1EC3", "U+1EC4", "U+1EC5", "U+1EC6", "U+1EC7", "U+1EC8",
                                        "U+1EC9", "U+1ECA", "U+1ECB", "U+1ECC", "U+1ECD", "U+1ECE", "U+1ECF",
                                        "U+1ED0", "U+1ED1", "U+1ED2", "U+1ED3", "U+1ED4", "U+1ED5", "U+1ED6",
                                        "U+1ED7", "U+1ED8", "U+1ED9", "U+1EDA", "U+1EDB", "U+1EDC", "U+1EDD",
                                        "U+1EDE", "U+1EDF", "U+1EE0", "U+1EE1", "U+1EE2", "U+1EE3", "U+1EE4",
                                        "U+1EE5", "U+1EE6", "U+1EE7", "U+1EE8", "U+1EE9", "U+1EEA", "U+1EEB",
                                        "U+1EEC", "U+1EED", "U+1EEE", "U+1EEF", "U+1EF0", "U+1EF1", "U+1EF2",
                                        "U+1EF3", "U+1EF4", "U+1EF5", "U+1EF6", "U+1EF7", "U+1EF8", "U+1EF9"
  ), VNI_Hex = c("41 D8", "41 D9", "41 C2", "41 D5", "45 D8", "45 D9",
                 "45 C2", "CC", "CD", "4F D8", "4F D9", "4F C2", "4F D5", "55 D8",
                 "55 D9", "59 D9", "61 F8", "61 F9", "61 E2", "61 F5", "65 F8",
                 "65 F9", "65 E2", "EC", "ED", "6F F8", "6F F9", "6F E2", "6F F5",
                 "75 F8", "75 F9", "79 F9", "41 CA", "61 EA", "D1", "F1", "D3",
                 "F3", "55 D5", "75 F5", "D4", "F4", "D6", "F6", "41 CF", "61 EF",
                 "41 DB", "61 FB", "41 C1", "61 E1", "41 C0", "61 E0", "41 C5",
                 "61 E5", "41 C3", "61 E3", "41 C4", "61 E4", "41 C9", "61 E9",
                 "41 C8", "61 E8", "41 DA", "61 FA", "41 DC", "61 FC", "41 CB",
                 "61 EB", "45 CF", "65 EF", "45 DB", "65 FB", "45 D5", "65 F5",
                 "45 C1", "65 E1", "45 C0", "65 E0", "45 C5", "65 E5", "45 C3",
                 "65 E3", "45 C4", "65 E4", "C6", "E6", "D2", "F2", "4F CF", "6F EF",
                 "4F DB", "6F FB", "4F C1", "6F E1", "4F C0", "6F E0", "4F C5",
                 "6F E5", "4F C3", "6F E3", "4F C4", "6F E4", "D4 D9", "F4 F9",
                 "D4 D8", "F4 F8", "D4 DB", "F4 FB", "D4 D5", "F4 F5", "D4 CF",
                 "F4 EF", "55 CF", "75 EF", "55 DB", "75 FB", "D6 D9", "F6 F9",
                 "D6 D8", "F6 F8", "D6 DB", "F6 FB", "D6 D5", "F6 F5", "D6 CF",
                 "F6 EF", "59 D8", "79 F8", "CE", "EE", "59 DB", "79 FB", "59 D5",
                 "79 F5"), VPS_Hex = c("80", "C1", "C2", "82", "D7", "C9", "CA",
                                       "B5", "B4", "BC", "B9", "D4", "BE", "A8", "DA", "DD", "E0", "E1",
                                       "E2", "E3", "E8", "E9", "EA", "EC", "ED", "F2", "F3", "F4", "F5",
                                       "F9", "FA", "9A", "88", "E6", "F1", "C7", "B8", "EF", "AC", "DB",
                                       "F7", "D6", "D0", "DC", "2", "E5", "81", "E4", "83", "C3", "84",
                                       "C0", "85", "C4", "1C", "C5", "3", "C6", "8D", "A1", "8E", "A2",
                                       "8F", "A3", "F0", "A4", "4", "A5", "5", "CB", "DE", "C8", "FE",
                                       "EB", "90", "89", "93", "8A", "94", "8B", "95", "CD", "6", "8C",
                                       "B7", "CC", "10", "CE", "11", "86", "BD", "D5", "96", "D3", "97",
                                       "D2", "98", "B0", "99", "87", "12", "B6", "9D", "A7", "9E", "A9",
                                       "9F", "AA", "A6", "AB", "13", "AE", "14", "F8", "D1", "FB", "AD",
                                       "D9", "AF", "D8", "B1", "BA", "1D", "BB", "15", "BF", "B2", "FF",
                                       "19", "9C", "FD", "9B", "B3", "CF"), VISCII_Hex = c("C0", "C1",
                                                                                           "C2", "C3", "C8", "C9", "CA", "CC", "CD", "D2", "D3", "D4", "A0",
                                                                                           "D9", "DA", "DD", "E0", "E1", "E2", "E3", "E8", "E9", "EA", "EC",
                                                                                           "ED", "F2", "F3", "F4", "F5", "F9", "FA", "FD", "C5", "E5", "D0",
                                                                                           "F0", "CE", "EE", "9D", "FB", "B4", "BD", "BF", "DF", "80", "D5",
                                                                                           "C4", "E4", "84", "A4", "85", "A5", "86", "A6", "6", "E7", "87",
                                                                                           "A7", "81", "A1", "82", "A2", "2", "C6", "5", "C7", "83", "A3",
                                                                                           "89", "A9", "CB", "EB", "88", "A8", "8A", "AA", "8B", "AB", "8C",
                                                                                           "AC", "8D", "AD", "8E", "AE", "9B", "EF", "98", "B8", "9A", "F7",
                                                                                           "99", "F6", "8F", "AF", "90", "B0", "91", "B1", "92", "B2", "93",
                                                                                           "B5", "95", "BE", "96", "B6", "97", "B7", "B3", "DE", "94", "FE",
                                                                                           "9E", "F8", "9C", "FC", "BA", "D1", "BB", "D7", "BC", "D8", "FF",
                                                                                           "E6", "B9", "F1", "9F", "CF", "1E", "DC", "14", "D6", "19", "DB"
                                       ), TCVN3_Hex = c("41 B5", "41 B8", "A2", "41 B7", "45 CC", "45 D0",
                                                        "A3", "49 D7", "49 DD", "4F DF", "4F E3", "A4", "4F E2", "55 EF",
                                                        "55 F3", "59 FD", "B5", "B8", "A9", "B7", "CC", "D0", "AA", "D7",
                                                        "DD", "DF", "E3", "AB", "E2", "EF", "F3", "FD", "A1", "A8", "A7",
                                                        "AE", "49 DC", "DC", "55 F2", "F2", "A5", "AC", "A6", "AD", "41 B9",
                                                        "B9", "41 B6", "B6", "A2 CA", "CA", "A2 C7", "C7", "A2 C8", "C8",
                                                        "A2 C9", "C9", "A2 CB", "CB", "A1 BE", "BE", "A1 BB", "BB", "A1 BC",
                                                        "BC", "A1 BD", "BD", "A1 C6", "C6", "45 D1", "D1", "45 CE", "CE",
                                                        "45 CF", "CF", "A3 D5", "D5", "A3 D2", "D2", "A3 D3", "D3", "A3 D4",
                                                        "D4", "A3 D6", "D6", "49 D8", "D8", "49 DE", "DE", "4F E4", "E4",
                                                        "4F E1", "E1", "A4 E8", "E8", "A4 E5", "E5", "A4 E6", "E6", "A4 E7",
                                                        "E7", "A4 E9", "E9", "A5 ED", "ED", "A5 EA", "EA", "A5 EB", "EB",
                                                        "A5 EC", "EC", "A5 EE", "EE", "55 F4", "F4", "55 F1", "F1", "A6 F8",
                                                        "F8", "A6 F5", "F5", "A6 F6", "F6", "A6 F7", "F7", "A6 F9", "F9",
                                                        "59 FA", "FA", "59 FE", "FE", "59 FB", "FB", "59 FC", "FC"),
  VIQR = c("A`", "A'", "A^", "A~", "E`", "E'", "E^", "I`",
           "I'", "O`", "O'", "O^", "O~", "U`", "U'", "Y'", "a`", "a'",
           "a^", "a~", "e`", "e'", "e^", "i`", "i'", "o`", "o'", "o^",
           "o~", "u`", "u'", "y'", "A(", "a(", "DD", "dd", "I~", "i~",
           "U~", "u~", "O+", "o+", "U+", "u+", "A.", "a.", "A?", "a?",
           "A^'", "a^'", "A^`", "a^`", "A^?", "a^?", "A^~", "a^~", "A^.",
           "a^.", "A('", "a('", "A(`", "a(`", "A(?", "a(?", "A(~", "a(~",
           "A(.", "a(.", "E.", "e.", "E?", "e?", "E~", "e~", "E^'",
           "e^'", "E^`", "e^`", "E^?", "e^?", "E^~", "e^~", "E^.", "e^.",
           "I?", "i?", "I.", "i.", "O.", "o.", "O?", "o?", "O^'", "o^'",
           "O^`", "o^`", "O^?", "o^?", "O^~", "o^~", "O^.", "o^.", "O+'",
           "o+'", "O+`", "o+`", "O+?", "o+?", "O+~", "o+~", "O+.", "o+.",
           "U.", "u.", "U?", "u?", "U+'", "u+'", "U+`", "u+`", "U+?",
           "u+?", "U+~", "u+~", "U+.", "u+.", "Y`", "y`", "Y.", "y.",
           "Y?", "y?", "Y~", "y~"), English_Name = c("LATIN CAPITAL LETTER A WITH GRAVE",
                                                     "LATIN CAPITAL LETTER A WITH ACUTE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX",
                                                     "LATIN CAPITAL LETTER A WITH TILDE", "LATIN CAPITAL LETTER E WITH GRAVE",
                                                     "LATIN CAPITAL LETTER E WITH ACUTE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX",
                                                     "LATIN CAPITAL LETTER I WITH GRAVE", "LATIN CAPITAL LETTER I WITH ACUTE",
                                                     "LATIN CAPITAL LETTER O WITH GRAVE", "LATIN CAPITAL LETTER O WITH ACUTE",
                                                     "LATIN CAPITAL LETTER O WITH CIRCUMFLEX", "LATIN CAPITAL LETTER O WITH TILDE",
                                                     "LATIN CAPITAL LETTER U WITH GRAVE", "LATIN CAPITAL LETTER U WITH ACUTE",
                                                     "LATIN CAPITAL LETTER Y WITH ACUTE", "LATIN SMALL LETTER A WITH GRAVE",
                                                     "LATIN SMALL LETTER A WITH ACUTE", "LATIN SMALL LETTER A WITH CIRCUMFLEX",
                                                     "LATIN SMALL LETTER A WITH TILDE", "LATIN SMALL LETTER E WITH GRAVE",
                                                     "LATIN SMALL LETTER E WITH ACUTE", "LATIN SMALL LETTER E WITH CIRCUMFLEX",
                                                     "LATIN SMALL LETTER I WITH GRAVE", "LATIN SMALL LETTER I WITH ACUTE",
                                                     "LATIN SMALL LETTER O WITH GRAVE", "LATIN SMALL LETTER O WITH ACUTE",
                                                     "LATIN SMALL LETTER O WITH CIRCUMFLEX", "LATIN SMALL LETTER O WITH TILDE",
                                                     "LATIN SMALL LETTER U WITH GRAVE", "LATIN SMALL LETTER U WITH ACUTE",
                                                     "LATIN SMALL LETTER Y WITH ACUTE", "LATIN CAPITAL LETTER A WITH BREVE",
                                                     "LATIN SMALL LETTER A WITH BREVE", "LATIN CAPITAL LETTER D WITH STROKE",
                                                     "LATIN SMALL LETTER D WITH STROKE", "LATIN CAPITAL LETTER I WITH TILDE",
                                                     "LATIN SMALL LETTER I WITH TILDE", "LATIN CAPITAL LETTER U WITH TILDE",
                                                     "LATIN SMALL LETTER U WITH TILDE", "LATIN CAPITAL LETTER O WITH HORN",
                                                     "LATIN SMALL LETTER O WITH HORN", "LATIN CAPITAL LETTER U WITH HORN",
                                                     "LATIN SMALL LETTER U WITH HORN", "LATIN CAPITAL LETTER A WITH DOT BELOW",
                                                     "LATIN SMALL LETTER A WITH DOT BELOW", "LATIN CAPITAL LETTER A WITH HOOK ABOVE",
                                                     "LATIN SMALL LETTER A WITH HOOK ABOVE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE",
                                                     "LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE",
                                                     "LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE",
                                                     "LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE",
                                                     "LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW",
                                                     "LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW", "LATIN CAPITAL LETTER A WITH BREVE AND ACUTE",
                                                     "LATIN SMALL LETTER A WITH BREVE AND ACUTE", "LATIN CAPITAL LETTER A WITH BREVE AND GRAVE",
                                                     "LATIN SMALL LETTER A WITH BREVE AND GRAVE", "LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE",
                                                     "LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE", "LATIN CAPITAL LETTER A WITH BREVE AND TILDE",
                                                     "LATIN SMALL LETTER A WITH BREVE AND TILDE", "LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW",
                                                     "LATIN SMALL LETTER A WITH BREVE AND DOT BELOW", "LATIN CAPITAL LETTER E WITH DOT BELOW",
                                                     "LATIN SMALL LETTER E WITH DOT BELOW", "LATIN CAPITAL LETTER E WITH HOOK ABOVE",
                                                     "LATIN SMALL LETTER E WITH HOOK ABOVE", "LATIN CAPITAL LETTER E WITH TILDE",
                                                     "LATIN SMALL LETTER E WITH TILDE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE",
                                                     "LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE",
                                                     "LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE",
                                                     "LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE",
                                                     "LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW",
                                                     "LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW", "LATIN CAPITAL LETTER I WITH HOOK ABOVE",
                                                     "LATIN SMALL LETTER I WITH HOOK ABOVE", "LATIN CAPITAL LETTER I WITH DOT BELOW",
                                                     "LATIN SMALL LETTER I WITH DOT BELOW", "LATIN CAPITAL LETTER O WITH DOT BELOW",
                                                     "LATIN SMALL LETTER O WITH DOT BELOW", "LATIN CAPITAL LETTER O WITH HOOK ABOVE",
                                                     "LATIN SMALL LETTER O WITH HOOK ABOVE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE",
                                                     "LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE",
                                                     "LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE",
                                                     "LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE",
                                                     "LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW",
                                                     "LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW", "LATIN CAPITAL LETTER O WITH HORN AND ACUTE",
                                                     "LATIN SMALL LETTER O WITH HORN AND ACUTE", "LATIN CAPITAL LETTER O WITH HORN AND GRAVE",
                                                     "LATIN SMALL LETTER O WITH HORN AND GRAVE", "LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE",
                                                     "LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE", "LATIN CAPITAL LETTER O WITH HORN AND TILDE",
                                                     "LATIN SMALL LETTER O WITH HORN AND TILDE", "LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW",
                                                     "LATIN SMALL LETTER O WITH HORN AND DOT BELOW", "LATIN CAPITAL LETTER U WITH DOT BELOW",
                                                     "LATIN SMALL LETTER U WITH DOT BELOW", "LATIN CAPITAL LETTER U WITH HOOK ABOVE",
                                                     "LATIN SMALL LETTER U WITH HOOK ABOVE", "LATIN CAPITAL LETTER U WITH HORN AND ACUTE",
                                                     "LATIN SMALL LETTER U WITH HORN AND ACUTE", "LATIN CAPITAL LETTER U WITH HORN AND GRAVE",
                                                     "LATIN SMALL LETTER U WITH HORN AND GRAVE", "LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE",
                                                     "LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE", "LATIN CAPITAL LETTER U WITH HORN AND TILDE",
                                                     "LATIN SMALL LETTER U WITH HORN AND TILDE", "LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW",
                                                     "LATIN SMALL LETTER U WITH HORN AND DOT BELOW", "LATIN CAPITAL LETTER Y WITH GRAVE",
                                                     "LATIN SMALL LETTER Y WITH GRAVE", "LATIN CAPITAL LETTER Y WITH DOT BELOW",
                                                     "LATIN SMALL LETTER Y WITH DOT BELOW", "LATIN CAPITAL LETTER Y WITH HOOK ABOVE",
                                                     "LATIN SMALL LETTER Y WITH HOOK ABOVE", "LATIN CAPITAL LETTER Y WITH TILDE",
                                                     "LATIN SMALL LETTER Y WITH TILDE"), ascii = c("A", "A", "A",
                                                                                                   "A", "E", "E", "E", "I", "I", "O", "O", "O", "O", "U", "U",
                                                                                                   "Y", "a", "a", "a", "a", "e", "e", "e", "i", "i", "o", "o",
                                                                                                   "o", "o", "u", "u", "y", "A", "a", "D", "d", "I", "i", "U",
                                                                                                   "u", "O", "o", "U", "u", "A", "a", "A", "a", "A", "a", "A",
                                                                                                   "a", "A", "a", "A", "a", "A", "a", "A", "a", "A", "a", "A",
                                                                                                   "a", "A", "a", "A", "a", "E", "e", "E", "e", "E", "e", "E",
                                                                                                   "e", "E", "e", "E", "e", "E", "e", "E", "e", "I", "i", "I",
                                                                                                   "i", "O", "o", "O", "o", "O", "o", "O", "o", "O", "o", "O",
                                                                                                   "o", "O", "o", "O", "o", "O", "o", "O", "o", "O", "o", "O",
                                                                                                   "o", "U", "u", "U", "u", "U", "u", "U", "u", "U", "u", "U",
                                                                                                   "u", "U", "u", "Y", "y", "Y", "y", "Y", "y", "Y", "y")), row.names = c(NA,
                                                                                                                                                                          -134L), class = "data.frame")  #out[,1] <- as_utf8(out[,1])


  out$Unicode <- intToUtf8(gsub("U+", "0x", out$Unicode_Hex, fixed = T), multiple = TRUE)

  out$VNI <- sapply(strsplit(out$VNI_Hex, " "), FUN = function(x) {
    intToUtf8(paste0("0x", x))})

  out$VPS <- sapply(strsplit(out$VPS_Hex, " "), FUN = function(x) {
    intToUtf8(paste0("0x", x))})

  out$VISCII <- sapply(strsplit(out$VISCII_Hex, " "), FUN = function(x) {
    intToUtf8(paste0("0x", x))})

  out$TCVN3 <- sapply(strsplit(out$TCVN3_Hex, " "), FUN = function(x) {
    intToUtf8(paste0("0x", x))})

  }

  if(version == 2){

    out <- structure(list(Unicode_Hex = c("U+00C0", "U+00C1", "U+00C2",
                                          "U+00C3", "U+00C8", "U+00C9", "U+00CA", "U+00CC", "U+00CD", "U+00D2",
                                          "U+00D3", "U+00D4", "U+00D5", "U+00D9", "U+00DA", "U+00DD", "U+00E0",
                                          "U+00E1", "U+00E2", "U+00E3", "U+00E8", "U+00E9", "U+00EA", "U+00EC",
                                          "U+00ED", "U+00F2", "U+00F3", "U+00F4", "U+00F5", "U+00F9", "U+00FA",
                                          "U+00FD", "U+0102", "U+0103", "U+0110", "U+0111", "U+0128", "U+0129",
                                          "U+0168", "U+0169", "U+01A0", "U+01A1", "U+01AF", "U+01B0", "U+1EA0",
                                          "U+1EA1", "U+1EA2", "U+1EA3", "U+1EA4", "U+1EA5", "U+1EA6", "U+1EA7",
                                          "U+1EA8", "U+1EA9", "U+1EAA", "U+1EAB", "U+1EAC", "U+1EAD", "U+1EAE",
                                          "U+1EAF", "U+1EB0", "U+1EB1", "U+1EB2", "U+1EB3", "U+1EB4", "U+1EB5",
                                          "U+1EB6", "U+1EB7", "U+1EB8", "U+1EB9", "U+1EBA", "U+1EBB", "U+1EBC",
                                          "U+1EBD", "U+1EBE", "U+1EBF", "U+1EC0", "U+1EC1", "U+1EC2", "U+1EC3",
                                          "U+1EC4", "U+1EC5", "U+1EC6", "U+1EC7", "U+1EC8", "U+1EC9", "U+1ECA",
                                          "U+1ECB", "U+1ECC", "U+1ECD", "U+1ECE", "U+1ECF", "U+1ED0", "U+1ED1",
                                          "U+1ED2", "U+1ED3", "U+1ED4", "U+1ED5", "U+1ED6", "U+1ED7", "U+1ED8",
                                          "U+1ED9", "U+1EDA", "U+1EDB", "U+1EDC", "U+1EDD", "U+1EDE", "U+1EDF",
                                          "U+1EE0", "U+1EE1", "U+1EE2", "U+1EE3", "U+1EE4", "U+1EE5", "U+1EE6",
                                          "U+1EE7", "U+1EE8", "U+1EE9", "U+1EEA", "U+1EEB", "U+1EEC", "U+1EED",
                                          "U+1EEE", "U+1EEF", "U+1EF0", "U+1EF1", "U+1EF2", "U+1EF3", "U+1EF4",
                                          "U+1EF5", "U+1EF6", "U+1EF7", "U+1EF8", "U+1EF9"), ASCII = c("A",
                                                                                                       "A", "A", "A", "E", "E", "E", "I", "I", "O", "O", "O", "O", "U",
                                                                                                       "U", "Y", "a", "a", "a", "a", "e", "e", "e", "i", "i", "o", "o",
                                                                                                       "o", "o", "u", "u", "y", "A", "a", "D", "d", "I", "i", "U", "u",
                                                                                                       "O", "o", "U", "u", "A", "a", "A", "a", "A", "a", "A", "a", "A",
                                                                                                       "a", "A", "a", "A", "a", "A", "a", "A", "a", "A", "a", "A", "a",
                                                                                                       "A", "a", "E", "e", "E", "e", "E", "e", "E", "e", "E", "e", "E",
                                                                                                       "e", "E", "e", "E", "e", "I", "i", "I", "i", "O", "o", "O", "o",
                                                                                                       "O", "o", "O", "o", "O", "o", "O", "o", "O", "o", "O", "o", "O",
                                                                                                       "o", "O", "o", "O", "o", "O", "o", "U", "u", "U", "u", "U", "u",
                                                                                                       "U", "u", "U", "u", "U", "u", "U", "u", "Y", "y", "Y", "y", "Y",
                                                                                                       "y", "Y", "y"), English_Name = c("LATIN CAPITAL LETTER A WITH GRAVE",
                                                                                                                                        "LATIN CAPITAL LETTER A WITH ACUTE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX",
                                                                                                                                        "LATIN CAPITAL LETTER A WITH TILDE", "LATIN CAPITAL LETTER E WITH GRAVE",
                                                                                                                                        "LATIN CAPITAL LETTER E WITH ACUTE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX",
                                                                                                                                        "LATIN CAPITAL LETTER I WITH GRAVE", "LATIN CAPITAL LETTER I WITH ACUTE",
                                                                                                                                        "LATIN CAPITAL LETTER O WITH GRAVE", "LATIN CAPITAL LETTER O WITH ACUTE",
                                                                                                                                        "LATIN CAPITAL LETTER O WITH CIRCUMFLEX", "LATIN CAPITAL LETTER O WITH TILDE",
                                                                                                                                        "LATIN CAPITAL LETTER U WITH GRAVE", "LATIN CAPITAL LETTER U WITH ACUTE",
                                                                                                                                        "LATIN CAPITAL LETTER Y WITH ACUTE", "LATIN SMALL LETTER A WITH GRAVE",
                                                                                                                                        "LATIN SMALL LETTER A WITH ACUTE", "LATIN SMALL LETTER A WITH CIRCUMFLEX",
                                                                                                                                        "LATIN SMALL LETTER A WITH TILDE", "LATIN SMALL LETTER E WITH GRAVE",
                                                                                                                                        "LATIN SMALL LETTER E WITH ACUTE", "LATIN SMALL LETTER E WITH CIRCUMFLEX",
                                                                                                                                        "LATIN SMALL LETTER I WITH GRAVE", "LATIN SMALL LETTER I WITH ACUTE",
                                                                                                                                        "LATIN SMALL LETTER O WITH GRAVE", "LATIN SMALL LETTER O WITH ACUTE",
                                                                                                                                        "LATIN SMALL LETTER O WITH CIRCUMFLEX", "LATIN SMALL LETTER O WITH TILDE",
                                                                                                                                        "LATIN SMALL LETTER U WITH GRAVE", "LATIN SMALL LETTER U WITH ACUTE",
                                                                                                                                        "LATIN SMALL LETTER Y WITH ACUTE", "LATIN CAPITAL LETTER A WITH BREVE",
                                                                                                                                        "LATIN SMALL LETTER A WITH BREVE", "LATIN CAPITAL LETTER D WITH STROKE",
                                                                                                                                        "LATIN SMALL LETTER D WITH STROKE", "LATIN CAPITAL LETTER I WITH TILDE",
                                                                                                                                        "LATIN SMALL LETTER I WITH TILDE", "LATIN CAPITAL LETTER U WITH TILDE",
                                                                                                                                        "LATIN SMALL LETTER U WITH TILDE", "LATIN CAPITAL LETTER O WITH HORN",
                                                                                                                                        "LATIN SMALL LETTER O WITH HORN", "LATIN CAPITAL LETTER U WITH HORN",
                                                                                                                                        "LATIN SMALL LETTER U WITH HORN", "LATIN CAPITAL LETTER A WITH DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER A WITH DOT BELOW", "LATIN CAPITAL LETTER A WITH HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER A WITH HOOK ABOVE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE",
                                                                                                                                        "LATIN SMALL LETTER A WITH CIRCUMFLEX AND ACUTE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE",
                                                                                                                                        "LATIN SMALL LETTER A WITH CIRCUMFLEX AND GRAVE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE",
                                                                                                                                        "LATIN SMALL LETTER A WITH CIRCUMFLEX AND TILDE", "LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER A WITH CIRCUMFLEX AND DOT BELOW", "LATIN CAPITAL LETTER A WITH BREVE AND ACUTE",
                                                                                                                                        "LATIN SMALL LETTER A WITH BREVE AND ACUTE", "LATIN CAPITAL LETTER A WITH BREVE AND GRAVE",
                                                                                                                                        "LATIN SMALL LETTER A WITH BREVE AND GRAVE", "LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER A WITH BREVE AND HOOK ABOVE", "LATIN CAPITAL LETTER A WITH BREVE AND TILDE",
                                                                                                                                        "LATIN SMALL LETTER A WITH BREVE AND TILDE", "LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER A WITH BREVE AND DOT BELOW", "LATIN CAPITAL LETTER E WITH DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER E WITH DOT BELOW", "LATIN CAPITAL LETTER E WITH HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER E WITH HOOK ABOVE", "LATIN CAPITAL LETTER E WITH TILDE",
                                                                                                                                        "LATIN SMALL LETTER E WITH TILDE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE",
                                                                                                                                        "LATIN SMALL LETTER E WITH CIRCUMFLEX AND ACUTE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE",
                                                                                                                                        "LATIN SMALL LETTER E WITH CIRCUMFLEX AND GRAVE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE",
                                                                                                                                        "LATIN SMALL LETTER E WITH CIRCUMFLEX AND TILDE", "LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER E WITH CIRCUMFLEX AND DOT BELOW", "LATIN CAPITAL LETTER I WITH HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER I WITH HOOK ABOVE", "LATIN CAPITAL LETTER I WITH DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER I WITH DOT BELOW", "LATIN CAPITAL LETTER O WITH DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER O WITH DOT BELOW", "LATIN CAPITAL LETTER O WITH HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER O WITH HOOK ABOVE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE",
                                                                                                                                        "LATIN SMALL LETTER O WITH CIRCUMFLEX AND ACUTE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE",
                                                                                                                                        "LATIN SMALL LETTER O WITH CIRCUMFLEX AND GRAVE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE",
                                                                                                                                        "LATIN SMALL LETTER O WITH CIRCUMFLEX AND TILDE", "LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER O WITH CIRCUMFLEX AND DOT BELOW", "LATIN CAPITAL LETTER O WITH HORN AND ACUTE",
                                                                                                                                        "LATIN SMALL LETTER O WITH HORN AND ACUTE", "LATIN CAPITAL LETTER O WITH HORN AND GRAVE",
                                                                                                                                        "LATIN SMALL LETTER O WITH HORN AND GRAVE", "LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER O WITH HORN AND HOOK ABOVE", "LATIN CAPITAL LETTER O WITH HORN AND TILDE",
                                                                                                                                        "LATIN SMALL LETTER O WITH HORN AND TILDE", "LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER O WITH HORN AND DOT BELOW", "LATIN CAPITAL LETTER U WITH DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER U WITH DOT BELOW", "LATIN CAPITAL LETTER U WITH HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER U WITH HOOK ABOVE", "LATIN CAPITAL LETTER U WITH HORN AND ACUTE",
                                                                                                                                        "LATIN SMALL LETTER U WITH HORN AND ACUTE", "LATIN CAPITAL LETTER U WITH HORN AND GRAVE",
                                                                                                                                        "LATIN SMALL LETTER U WITH HORN AND GRAVE", "LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER U WITH HORN AND HOOK ABOVE", "LATIN CAPITAL LETTER U WITH HORN AND TILDE",
                                                                                                                                        "LATIN SMALL LETTER U WITH HORN AND TILDE", "LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER U WITH HORN AND DOT BELOW", "LATIN CAPITAL LETTER Y WITH GRAVE",
                                                                                                                                        "LATIN SMALL LETTER Y WITH GRAVE", "LATIN CAPITAL LETTER Y WITH DOT BELOW",
                                                                                                                                        "LATIN SMALL LETTER Y WITH DOT BELOW", "LATIN CAPITAL LETTER Y WITH HOOK ABOVE",
                                                                                                                                        "LATIN SMALL LETTER Y WITH HOOK ABOVE", "LATIN CAPITAL LETTER Y WITH TILDE",
                                                                                                                                        "LATIN SMALL LETTER Y WITH TILDE"), Unicode_int = c(192L, 193L,
                                                                                                                                                                                            194L, 195L, 200L, 201L, 202L, 204L, 205L, 210L, 211L, 212L, 213L,
                                                                                                                                                                                            217L, 218L, 221L, 224L, 225L, 226L, 227L, 232L, 233L, 234L, 236L,
                                                                                                                                                                                            237L, 242L, 243L, 244L, 245L, 249L, 250L, 253L, 258L, 259L, 208L,
                                                                                                                                                                                            273L, 296L, 297L, 360L, 361L, 416L, 417L, 431L, 432L, 7840L,
                                                                                                                                                                                            7841L, 7842L, 7843L, 7844L, 7845L, 7846L, 7847L, 7848L, 7849L,
                                                                                                                                                                                            7850L, 7851L, 7852L, 7853L, 7854L, 7855L, 7856L, 7857L, 7858L,
                                                                                                                                                                                            7859L, 7860L, 7861L, 7862L, 7863L, 7864L, 7865L, 7866L, 7867L,
                                                                                                                                                                                            7868L, 7869L, 7870L, 7871L, 7872L, 7873L, 7874L, 7875L, 7876L,
                                                                                                                                                                                            7877L, 7878L, 7879L, 7880L, 7881L, 7882L, 7883L, 7884L, 7885L,
                                                                                                                                                                                            7886L, 7887L, 7888L, 7889L, 7890L, 7891L, 7892L, 7893L, 7894L,
                                                                                                                                                                                            7895L, 7896L, 7897L, 7898L, 7899L, 7900L, 7901L, 7902L, 7903L,
                                                                                                                                                                                            7904L, 7905L, 7906L, 7907L, 7908L, 7909L, 7910L, 7911L, 7912L,
                                                                                                                                                                                            7913L, 7914L, 7915L, 7916L, 7917L, 7918L, 7919L, 7920L, 7921L,
                                                                                                                                                                                            7922L, 7923L, 7924L, 7925L, 7926L, 7927L, 7928L, 7929L), TCVN3_int = list(
                                                                                                                                                                                              c(65L, 181L), c(65L, 184L), 162L, c(65L, 183L), c(69L, 204L
                                                                                                                                                                                              ), c(69L, 208L), 163L, c(73L, 215L), c(73L, 221L), c(79L,
                                                                                                                                                                                                                                                   223L), c(79L, 227L), 164L, c(79L, 226L), c(85L, 239L), c(85L,
                                                                                                                                                                                                                                                                                                            243L), c(89L, 253L), 181L, 184L, 169L, 183L, 204L, 208L,
                                                                                                                                                                                              170L, 215L, 221L, 223L, 227L, 171L, 226L, 239L, 243L, 253L,
                                                                                                                                                                                              161L, 168L, 167L, 174L, c(73L, 220L), 220L, c(85L, 242L),
                                                                                                                                                                                              242L, 165L, 172L, 166L, 173L, c(65L, 185L), 185L, c(65L,
                                                                                                                                                                                                                                                  182L), 182L, c(162L, 202L), 202L, c(162L, 199L), 199L, c(162L,
                                                                                                                                                                                                                                                                                                           200L), 200L, c(162L, 201L), 201L, c(162L, 203L), 203L, c(161L,
                                                                                                                                                                                                                                                                                                                                                                    190L), 190L, c(161L, 187L), 187L, c(161L, 188L), 188L, c(161L,
                                                                                                                                                                                                                                                                                                                                                                                                                             189L), 189L, c(161L, 198L), 198L, c(69L, 209L), 209L, c(69L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     206L), 206L, c(69L, 207L), 207L, c(163L, 213L), 213L, c(163L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             210L), 210L, c(163L, 211L), 211L, c(163L, 212L), 212L, c(163L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      214L), 214L, c(73L, 216L), 216L, c(73L, 222L), 222L, c(79L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             228L), 228L, c(79L, 225L), 225L, c(164L, 232L), 232L, c(164L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     229L), 229L, c(164L, 230L), 230L, c(164L, 231L), 231L, c(164L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              233L), 233L, c(165L, 237L), 237L, c(165L, 234L), 234L, c(165L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       235L), 235L, c(165L, 236L), 236L, c(165L, 238L), 238L, c(85L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                244L), 244L, c(85L, 241L), 241L, c(166L, 248L), 248L, c(166L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        245L), 245L, c(166L, 246L), 246L, c(166L, 247L), 247L, c(166L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 249L), 249L, c(89L, 250L), 250L, c(89L, 254L), 254L, c(89L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        251L), 251L, c(89L, 252L), 252L), VISCII_int = c(192L, 193L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         194L, 195L, 200L, 201L, 202L, 204L, 205L, 210L, 211L, 212L, NA,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         217L, 218L, 221L, 224L, 225L, 226L, 227L, 232L, 233L, 234L, 236L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         237L, 242L, 243L, 244L, 245L, 249L, 250L, 253L, 197L, 229L, 208L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         240L, 206L, 238L, 157L, 251L, 180L, 189L, 191L, 223L, 8364L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         213L, 196L, 228L, 8222L, 164L, 8230L, 165L, 8224L, 166L, 6L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         231L, 8225L, 167L, 129L, 161L, 8218L, 162L, 2L, 198L, 5L, 199L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         402L, 163L, 8240L, 169L, 203L, 235L, 710L, 168L, 352L, 170L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         8249L, 171L, 338L, 172L, 141L, 173L, 381L, 174L, 8250L, 239L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         732L, 184L, 353L, 247L, 8482L, 246L, 143L, 175L, 144L, 176L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         8216L, 177L, 8217L, 178L, 8220L, 181L, 8226L, 190L, 8211L, 182L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         8212L, 183L, 179L, 222L, 8221L, 254L, 382L, 248L, 339L, 252L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         186L, 209L, 187L, 215L, 188L, 216L, 255L, 230L, 185L, 241L, 376L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         207L, 30L, 220L, 20L, 214L, 25L, 219L), VNI_int = list(c(65L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  216L), c(65L, 217L), c(65L, 194L), c(65L, 213L), c(69L, 216L),
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                c(69L, 217L), c(69L, 194L), 204L, 205L, c(79L, 216L), c(79L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        217L), c(79L, 194L), c(79L, 213L), c(85L, 216L), c(85L, 217L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        ), c(89L, 217L), c(97L, 248L), c(97L, 249L), c(97L, 226L),
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                c(97L, 245L), c(101L, 248L), c(101L, 249L), c(101L, 226L),
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                236L, 237L, c(111L, 248L), c(111L, 249L), c(111L, 226L),
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                c(111L, 245L), c(117L, 248L), c(117L, 249L), c(121L, 249L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                ), c(65L, 202L), c(97L, 234L), 209L, 241L, 211L, 243L, c(85L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         213L), c(117L, 245L), 212L, 244L, 214L, 246L, c(65L, 207L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         ), c(97L, 239L), c(65L, 219L), c(97L, 251L), c(65L, 193L),
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                c(97L, 225L), c(65L, 192L), c(97L, 224L), c(65L, 197L), c(97L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          229L), c(65L, 195L), c(97L, 227L), c(65L, 196L), c(97L, 228L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          ), c(65L, 201L), c(97L, 233L), c(65L, 200L), c(97L, 232L),
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                c(65L, 218L), c(97L, 250L), c(65L, 220L), c(97L, 252L), c(65L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          203L), c(97L, 235L), c(69L, 207L), c(101L, 239L), c(69L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              219L), c(101L, 251L), c(69L, 213L), c(101L, 245L), c(69L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   193L), c(101L, 225L), c(69L, 192L), c(101L, 224L), c(69L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        197L), c(101L, 229L), c(69L, 195L), c(101L, 227L), c(69L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             196L), c(101L, 228L), 198L, 230L, 210L, 242L, c(79L, 207L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ), c(111L, 239L), c(79L, 219L), c(111L, 251L), c(79L, 193L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ), c(111L, 225L), c(79L, 192L), c(111L, 224L), c(79L, 197L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ), c(111L, 229L), c(79L, 195L), c(111L, 227L), c(79L, 196L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ), c(111L, 228L), c(212L, 217L), c(244L, 249L), c(212L, 216L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             ), c(244L, 248L), c(212L, 219L), c(244L, 251L), 212:213,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                244:245, c(212L, 207L), c(244L, 239L), c(85L, 207L), c(117L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       239L), c(85L, 219L), c(117L, 251L), c(214L, 217L), c(246L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            249L), c(214L, 216L), c(246L, 248L), c(214L, 219L), c(246L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  251L), 214:213, 246:245, c(214L, 207L), c(246L, 239L), c(89L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           216L), c(121L, 248L), 206L, 238L, c(89L, 219L), c(121L, 251L
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           ), c(89L, 213L), c(121L, 245L)), VPS_int = c(8364L, 193L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        194L, 8218L, 215L, 201L, 202L, 181L, 180L, 188L, 185L, 212L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        190L, 168L, 218L, 221L, 224L, 225L, 226L, 227L, 232L, 233L, 234L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        236L, 237L, 242L, 243L, 244L, 245L, 249L, 250L, 353L, 710L, 230L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        241L, 199L, 184L, 239L, 172L, 219L, 247L, 214L, 208L, 220L, 2L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        229L, 129L, 228L, 402L, 195L, 8222L, 192L, 8230L, 196L, 28L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        197L, 3L, 198L, 141L, 161L, 381L, 162L, 143L, 163L, 240L, 164L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        4L, 165L, 5L, 203L, 222L, 200L, 254L, 235L, 144L, 8240L, 8220L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        352L, 8221L, 8249L, 8226L, 205L, 6L, 338L, 183L, 204L, 16L, 206L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        17L, 8224L, 189L, 213L, 8211L, 211L, 8212L, 210L, 732L, 176L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        8482L, 8225L, 18L, 182L, 157L, 167L, 382L, 169L, 376L, 170L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        166L, 171L, 19L, 174L, 20L, 248L, 209L, 251L, 173L, 217L, 175L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        216L, 177L, 186L, 29L, 187L, 21L, 191L, 178L, 255L, 25L, 339L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        253L, 8250L, 179L, 207L), VNU_int = c(129L, 8364L, 8222L, 195L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              200L, 201L, 8240L, 204L, 205L, 210L, 338L, 141L, 213L, 217L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              732L, 382L, 161L, 376L, 181L, 172L, 191L, 190L, 197L, 214L, 207L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              221L, 220L, 225L, 223L, 239L, 238L, 251L, 402L, 175L, 8225L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              189L, 296L, 216L, 360L, 241L, 381L, 231L, 339L, 245L, 7840L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              173L, 8218L, 168L, 8230L, 182L, 7846L, 183L, 7848L, 184L, 6L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              185L, 7852L, 186L, 7854L, 176L, 7856L, 177L, 7858L, 178L, 7860L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              179L, 7862L, 180L, 7864L, 194L, 7866L, 192L, 7868L, 193L, 7870L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              198L, 7872L, 203L, 7874L, 204L, 7876L, 205L, 7878L, 206L, 7880L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              215L, 7882L, 217L, 7884L, 224L, 7886L, 222L, 7888L, 226L, 7890L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              227L, 7892L, 228L, 7894L, 229L, 7896L, 230L, 7898L, 232L, 7900L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              233L, 143L, 234L, 7904L, 235L, 7906L, 236L, 7908L, 242L, 8482L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              240L, 157L, 246L, 7914L, 247L, 7916L, 248L, 7918L, 249L, 7920L,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              250L, 7922L, 252L, 7924L, 255L, 7926L, 253L, 7928L, 254L)), row.names = c(NA,
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        -134L), class = "data.frame")

    out$Unicode <- sapply(out$Unicode_int, intToUtf8)
    out$TCVN3 <- sapply(out$TCVN3_int, intToUtf8)
    out$VISCII <- sapply(out$VISCII_int, intToUtf8)
    out$VNI <- sapply(out$VNI_int, intToUtf8)
    out$VPS <- sapply(out$VPS_int, intToUtf8)
    out$VNU <- sapply(out$VNU_int, intToUtf8)

    }
  return(out)
}

