% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/racusum_mc.R
\name{racusum_arl_mc}
\alias{racusum_arl_mc}
\title{Compute ARLs of RA-CUSUM control charts using Markov chain approximation}
\usage{
racusum_arl_mc(pmix, RA, RQ, h, scaling = 600, rounding = "p", method = "Toep")
}
\arguments{
\item{pmix}{Numeric Matrix. A three column matrix. First column is the risk
score distribution. Second column are the predicted probabilities from the risk model. Third
 column can be either the predicted probabilities from the risk model or average outcome per
 risk score, see examples.}

\item{RA}{Double. Odds ratio of death under the alternative hypotheses. Detecting deterioration
in performance with increased mortality risk by doubling the odds Ratio \code{RA = 2}. Detecting
 improvement in performance with decreased mortality risk by halving the odds ratio of death
 \code{RA = 1/2}. Odds ratio of death under the null hypotheses is \code{1}.}

\item{RQ}{Double. Defines the true performance of a surgeon with the odds ratio ratio of death
\code{RQ}. Use \code{RQ = 1} to compute the in-control ARL and other values to compute the
out-of-control ARL.}

\item{h}{Double. \code{h} is the control limit (>\code{0}).}

\item{scaling}{Double. The \code{scaling} parameter controls the quality of the approximation,
larger values achieve higher accuracy but increase the computation burden (larger transition
probability matrix).}

\item{rounding}{Character. If \code{rounding = "p"} a paired rounding implementation of
\emph{Knoth et al. (2019)} is used, if \code{rounding = "s"} a simple rounding method of
\emph{Steiner et al. (2000)} is used.}

\item{method}{Character. If \code{method = "Toep"} a combination of Sequential Probability Ratio
Test and Toeplitz matrix structure is used to calculate the ARL. \code{"ToepInv"} computes the
inverted matrix using Toeplitz matrix structure. \code{"BE"} solves a linear equation system
using the classical approach of \emph{Brook and Evans (1972)} to calculate the ARL.}
}
\value{
Returns a single value which is the Average Run Length.
}
\description{
Computes the Average Run Length of a risk-adjusted cumulative sum control chart
using Markov chain approximation.
}
\examples{
\dontrun{
library(vlad)
library(dplyr)
data("cardiacsurgery", package = "spcadjust")

## preprocess data to 30 day mortality and subset phase I (In-control) of surgeons 2
SALLI <- cardiacsurgery \%>\% rename(s = Parsonnet) \%>\%
  mutate(y = ifelse(status == 1 & time <= 30, 1, 0),
        phase = factor(ifelse(date < 2*365, "I", "II"))) \%>\%
  filter(phase == "I") \%>\% select(s, y)

## estimate risk model, get relative frequences and probabilities
mod1 <- glm(y ~ s, data = SALLI, family = "binomial")
fi  <- as.numeric(table(SALLI$s) / length(SALLI$s))
usi <- sort(unique(SALLI$s))
pi1 <- predict(mod1, newdata = data.frame(s = usi), type = "response")
pi2 <- tapply(SALLI$y, SALLI$s, mean)

## set up patient mix (risk model)
pmix1  <- data.frame(fi, pi1, pi1)

## Average Run Length for detecting deterioration RA = 2:
racusum_arl_mc(pmix = pmix1, RA = 2, RQ = 1, h = 4.5)

## Average Run Length for detecting improvement RA = 1/2:
racusum_arl_mc(pmix = pmix1, RA = 1/2, RQ = 1, h = 4)

## set up patient mix (model free)
pmix2  <- data.frame(fi, pi1, pi2)

## Average Run Length for detecting deterioration RA = 2:
racusum_arl_mc(pmix = pmix2, RA = 2, RQ = 1, h = 4.5)

## Average Run Length for detecting improvement RA = 1/2:
racusum_arl_mc(pmix = pmix2, RA = 1/2, RQ = 1, h = 4)

## compare results with R-code function 'findarl()' from Steiner et al. (2000)
source("https://bit.ly/2KC0SYD")
all.equal(findarl(pmix = pmix1, R1 = 2, R = 1, CL = 4.5, scaling = 600),
         racusum_arl_mc(pmix = pmix1, RA = 2, RQ = 1, h = 4.5, scaling = 600, rounding = "s"))
}
}
\references{
Knoth S, Wittenberg P and Gan FF (2019).
Risk-adjusted CUSUM charts under model error.
\emph{Statistics in Medicine}, \strong{38}(12), pp. 2206--2218.

Steiner SH, Cook RJ, Farewell VT and Treasure T (2000).
 Monitoring surgical performance using risk-adjusted cumulative sum charts.
 \emph{Biostatistics}, \strong{1}(4), pp. 441--452.

Brook D and Evans DA (1972)
 An approach to the probability distribution of CUSUM run length.
 \emph{Biometrika}, \strong{59}(3), pp. 539--549

Webster RA and Pettitt AN (2007)
Stability of approximations of average run length of risk-adjusted CUSUM schemes using
the Markov approach: comparing two methods of calculating transition probabilities.
 \emph{Communications in Statistics - Simulation and Computation} \strong{36}(3), pp. 471--482
}
\author{
Philipp Wittenberg
}
