#
# (c) 2021 Andreas Geyer-Schulz
#          Simple Genetic Algorithm in R. V 0.1
#          Layer: Population-level functions.
#                 Independent of gene representation.
#                 The replication mechanism and its variants
#          Package: xegaPopulation.
#

#' Replicates a gene (differential evolution).
#'
#' @description \code{ReplicateGeneDE} replicates a gene. Replication
#'              is the reproduction function which uses crossover and
#'              mutation. The control flow of differential evolution 
#'              is as follows:
#'              \itemize{
#'              \item A target gene is selected from the population.
#'              \item A mutant gene is generated by differential mutation.
#'              \item The gene and the mutant gene are crossed to get a 
#'                    new gene.
#'              \item The gene is accepted if it is at least as good 
#'                    as the target gene.
#'              }
#'
#' @details For \code{selection="UniformP"},
#'          for \code{crossover="UPCrossGene"} and
#'          for \code{accept="Best"}
#'          this is 
#'          the algorithm of Price, Storn and Lampinen (2005), page 41.
#'
#' @param pop    Population of binary genes.
#' @param fit    Fitness vector.
#' @param lF     Local configuration of the genetic algorithm.
#'
#' @return A list of one gene.
#'
#' @family Replication
#'
#' @references
#' Price, Kenneth V., Storn, Rainer M. and Lampinen, Jouni A. (2005)
#' The Differential Evolution Algorithm (Chapter 2), pp. 37-134.
#' In: Differential Evolution. A Practical Approach to Global Optimization.
#' Springer, Berlin.
#' <doi:10.1007/3-540-31306-0>
#'
#' @examples
#' pop10<-lapply(rep(0,10), function(x) xegaDfGene::xegaDfInitGene(lFxegaDfGene))
#' epop10<-lapply(pop10, lFxegaDfGene$EvalGene, lF=lFxegaDfGene)
#' fit10<-unlist(lapply(epop10, function(x) {x$fit}))
#' newgenes<-xegaDfReplicateGeneDE(pop10, fit10, lFxegaDfGene)
#' @importFrom xegaSelectGene parm
#' @export
xegaDfReplicateGeneDE<- function(pop, fit, lF)
{
targetGene<-pop[[lF$SelectGene(fit, lF)]]
gene0<-pop[[lF$SelectGene(fit, lF)]]
gene1<-pop[[lF$SelectGene(fit, lF)]]
gene2<-pop[[lF$SelectGene(fit, lF)]]
trialGene<-lF$CrossGene(targetGene,lF$MutateGene(gene0, gene1, gene2, lF))[[1]]
t1<-lF$EvalGene(trialGene, lF)
    lF$trialGene<-parm(t1)
    OperatorPipeline<-function(g, lF) {lF$trialGene()}   
    return(list(lF$Accept(OperatorPipeline, targetGene, lF)))
}

#' Configure the replication function of a genetic algorithm.
#'
#' @description \code{ReplicationFactory} implements the selection
#'              of a replication method. 
#'
#'              Current support:
#'
#'              \enumerate{
#'              \item "DE" returns \code{ReplicateGeneDE}.
#'              }
#'
#' @param method A string specifying the replication function.
#'
#' @return A replication function for genes.
#'
#' @family Configuration
#'
#' @examples
#' pop10<-lapply(rep(0,10), function(x) xegaDfInitGene(lFxegaDfGene))
#' epop10<-lapply(pop10, lFxegaDfGene$EvalGene, lF=lFxegaDfGene)
#' fit10<-unlist(lapply(epop10, function(x) {x$fit}))
#' Replicate<-xegaDfReplicationFactory("DE")
#' newgenes2<-Replicate(pop10, fit10, lFxegaDfGene)
#' @export
xegaDfReplicationFactory<-function(method="DE") {
if (method=="DE") {f<- xegaDfReplicateGeneDE}
if (!exists("f", inherits=FALSE))
        {stop("sgde Replication label ", method, " does not exist")}
return(f)
}

