


















use crate::RngCore;
use core::cmp::min;


pub fn next_u64_via_u32<R: RngCore + ?Sized>(rng: &mut R) -> u64 {

    let x = u64::from(rng.next_u32());
    let y = u64::from(rng.next_u32());
    (y << 32) | x
}







pub fn fill_bytes_via_next<R: RngCore + ?Sized>(rng: &mut R, dest: &mut [u8]) {
    let mut left = dest;
    while left.len() >= 8 {
        let (l, r) = { left }.split_at_mut(8);
        left = r;
        let chunk: [u8; 8] = rng.next_u64().to_le_bytes();
        l.copy_from_slice(&chunk);
    }
    let n = left.len();
    if n > 4 {
        let chunk: [u8; 8] = rng.next_u64().to_le_bytes();
        left.copy_from_slice(&chunk[..n]);
    } else if n > 0 {
        let chunk: [u8; 4] = rng.next_u32().to_le_bytes();
        left.copy_from_slice(&chunk[..n]);
    }
}

trait Observable: Copy {
    type Bytes: AsRef<[u8]>;
    fn to_le_bytes(self) -> Self::Bytes;


    fn as_byte_slice(x: &[Self]) -> &[u8];
}
impl Observable for u32 {
    type Bytes = [u8; 4];
    fn to_le_bytes(self) -> Self::Bytes {
        self.to_le_bytes()
    }
    fn as_byte_slice(x: &[Self]) -> &[u8] {
        let ptr = x.as_ptr() as *const u8;
        let len = x.len() * core::mem::size_of::<Self>();
        unsafe { core::slice::from_raw_parts(ptr, len) }
    }
}
impl Observable for u64 {
    type Bytes = [u8; 8];
    fn to_le_bytes(self) -> Self::Bytes {
        self.to_le_bytes()
    }
    fn as_byte_slice(x: &[Self]) -> &[u8] {
        let ptr = x.as_ptr() as *const u8;
        let len = x.len() * core::mem::size_of::<Self>();
        unsafe { core::slice::from_raw_parts(ptr, len) }
    }
}

fn fill_via_chunks<T: Observable>(src: &[T], dest: &mut [u8]) -> (usize, usize) {
    let size = core::mem::size_of::<T>();
    let byte_len = min(src.len() * size, dest.len());
    let num_chunks = (byte_len + size - 1) / size;

    if cfg!(target_endian = "little") {

        dest[..byte_len].copy_from_slice(&T::as_byte_slice(&src[..num_chunks])[..byte_len]);
    } else {

        let mut i = 0;
        let mut iter = dest[..byte_len].chunks_exact_mut(size);
        for chunk in &mut iter {
            chunk.copy_from_slice(src[i].to_le_bytes().as_ref());
            i += 1;
        }
        let chunk = iter.into_remainder();
        if !chunk.is_empty() {
            chunk.copy_from_slice(&src[i].to_le_bytes().as_ref()[..chunk.len()]);
        }
    }

    (num_chunks, byte_len)
}































pub fn fill_via_u32_chunks(src: &[u32], dest: &mut [u8]) -> (usize, usize) {
    fill_via_chunks(src, dest)
}











pub fn fill_via_u64_chunks(src: &[u64], dest: &mut [u8]) -> (usize, usize) {
    fill_via_chunks(src, dest)
}


pub fn next_u32_via_fill<R: RngCore + ?Sized>(rng: &mut R) -> u32 {
    let mut buf = [0; 4];
    rng.fill_bytes(&mut buf);
    u32::from_le_bytes(buf)
}


pub fn next_u64_via_fill<R: RngCore + ?Sized>(rng: &mut R) -> u64 {
    let mut buf = [0; 8];
    rng.fill_bytes(&mut buf);
    u64::from_le_bytes(buf)
}

#[cfg(test)]
mod test {
    use super::*;

    #[test]
    fn test_fill_via_u32_chunks() {
        let src = [1, 2, 3];
        let mut dst = [0u8; 11];
        assert_eq!(fill_via_u32_chunks(&src, &mut dst), (3, 11));
        assert_eq!(dst, [1, 0, 0, 0, 2, 0, 0, 0, 3, 0, 0]);

        let mut dst = [0u8; 13];
        assert_eq!(fill_via_u32_chunks(&src, &mut dst), (3, 12));
        assert_eq!(dst, [1, 0, 0, 0, 2, 0, 0, 0, 3, 0, 0, 0, 0]);

        let mut dst = [0u8; 5];
        assert_eq!(fill_via_u32_chunks(&src, &mut dst), (2, 5));
        assert_eq!(dst, [1, 0, 0, 0, 2]);
    }

    #[test]
    fn test_fill_via_u64_chunks() {
        let src = [1, 2];
        let mut dst = [0u8; 11];
        assert_eq!(fill_via_u64_chunks(&src, &mut dst), (2, 11));
        assert_eq!(dst, [1, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0]);

        let mut dst = [0u8; 17];
        assert_eq!(fill_via_u64_chunks(&src, &mut dst), (2, 16));
        assert_eq!(dst, [1, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0]);

        let mut dst = [0u8; 5];
        assert_eq!(fill_via_u64_chunks(&src, &mut dst), (1, 5));
        assert_eq!(dst, [1, 0, 0, 0, 0]);
    }
}
