% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions_misc.R
\name{coefs.start}
\alias{coefs.start}
\title{Initial values for coefficients}
\usage{
coefs.start(model, yt, y.start = NULL, y.lower = 0, y.upper = 1,
  xreg = NULL, p = 0, q = 0, d = FALSE, map = .default.map.barc,
  lags = NULL, fixed.values = NULL, fixed.lags = NULL,
  linkg = "linear", configs.linkg = NULL)
}
\arguments{
\item{model}{character string (case-insensitive) indicating the model to be
fitted to the data. Must be one of the options listed in the Section
\link[=arguments.model]{Supported Models}.}

\item{yt}{numeric vector with the observed time series. Missing values (NA's)
are not allowed.}

\item{y.start}{optional; an initial value for \eqn{Y_t} (to initialize
recursions when \eqn{t < 1}). Default is \code{y.start = NULL}, in which case,
the recursion assumes that \eqn{Y_t = g_{12}^{-1}(0)}, for \eqn{t < 1}.
Only relevant if \eqn{p > 0}.}

\item{y.lower}{the lower limit for the Kumaraswamy density support. Default
is \code{y.lower = 0}.}

\item{y.upper}{the upper limit for the Kumaraswamy density support. Default
is \code{y.upper = 1}.}

\item{xreg}{optional; external regressors. Can be specified as a vector, a
matrix or a list. Default is \code{xreg = NULL}. For details, see the Section
\link[=arguments.regressors]{Regressors format}.}

\item{p}{the AR order. Default is \code{p = 0}. Can be provided as either a single
integer (legacy format) or a length 2 integer vector (new format)
specifying orders for \code{part1}/\code{part2}. If \eqn{\nu} is time-varying and a
single value is provided it is assumed that \eqn{p_1 = p_2 = p}.}

\item{q}{the MA order. Default is \code{q = 0}. Can be provided as either a single
integer (legacy format) or a length 2 integer vector (new format)
specifying orders for \code{part1}/\code{part2}. If \eqn{\nu} is time-varying and a
single value is provided it is assumed that \eqn{q_1 = q_2 = q}.}

\item{d}{a length 1 (legacy format) or 2 (new format) logical vector
indicating whether the long memory parameter \eqn{d} is presented in the
model (either as a fixed or non-fixed parameter). In the new format, if
only one value is provided the code assumes that the same option is valid
for both parts of the model. Default is \code{d = FALSE}.}

\item{map}{a non-negative integer from 1 to 5 corresponding to the map
function. Default is \code{map = 4}. See the Section \link[=arguments.map]{The map function}.}

\item{lags}{optional; specification of which lags to include in the model.
For one dimensional coefficients, the lag is obviously always 1 and can
be suppressed. Can be specified in one of two ways
\itemize{
\item a list with components \code{beta}, \code{phi}, and \code{theta} (legacy format)
specifying which lags to include for each parameter type.

\item a list with elements \code{part1} and \code{part2} (new format), each being a
list with components \code{beta}, \code{phi}, and \code{theta} specifying which lags to
include for each parameter type.
}
Default is \code{lags = NULL}, in which the \code{lags} are computed from the
\code{fixed.lags} argument (if provided). When components are missing or empty
in both, \code{lags} and \code{fixed.lags}, the default behavior is to include all
lags based on \code{nreg = ncol(xreg)}, \code{p}, and \code{q}. The arguments \code{lags} and
\code{fixed.lags} are complementary. Either suffices, or mix them (e.g., \code{lags}
for some parameters, \code{fixed.lags} for others).}

\item{fixed.values}{optional; specification of fixed parameter values.
Can be specified in one of two ways
\itemize{
\item a list with optional components \code{alpha}, \code{beta}, \code{phi}, \code{theta}, \code{d}
and \code{nu} (legacy format) containing fixed values for each parameter type.

\item a list with elements \code{part1} and \code{part2} (new format), each being a
list with optional components \code{alpha}, \code{beta}, \code{phi}, \code{theta} and \code{d}
containing fixed values for each parameter type.
}
If fixed values are provided and there exists more than one possible lag,
either \code{lags} or \code{fixed.lags} must also be provided. The default is
\code{fixed.lags = NULL}. By default, if a given vector (such as the vector of
AR coefficients) has fixed values and the corresponding entry in this list
is empty, the fixed values are set as zero.}

\item{fixed.lags}{optional; specification of which lags should be fixed.
For one dimensional coefficients, the lag is obviously always 1 and can
be suppressed. Can be specified in one of two ways
\itemize{
\item a list with components \code{beta}, \code{phi}, and \code{theta} (legacy format)
specifying which lags should be fixed.

\item a list with elements \code{part1} and \code{part2} (new format), each being a
list with components \code{beta}, \code{phi}, and \code{theta} specifying which lags
should be fixed.
}
For missing components, fixed values will are set based on \code{lags}.}

\item{linkg}{specification of link functions. Can be specified in one of two
ways
\itemize{
\item A character or two-character vector (legacy format). If only one
string is provided, the same link name is used for \eqn{\mu_t}
(\eqn{g_{11}}) and for \eqn{Y_t} (\eqn{g_{12}}).

\item A named list (new format) with elements \code{g11}, \code{g12}, \code{g2}, \code{g21},
and \code{g22} (order does not matter). Particular models (see
\sQuote{Particular Models} in \link{BTSR.functions}) have default values for
some links. Missing links follow these rules
\itemize{
\item If either \code{g11} or \code{g12} is missing (but not both), assumes \code{g12 = g11}
\item If \code{phi = NULL} for part 1, \code{g12} is not required
\item If \code{phi = NULL} for part 2, \code{g22} is not required
\item If either \code{g21} or \code{g22} is missing (but not both), assumes \code{g22 = g21}
}
Special case: \code{g2 = "default"} uses the model's default link. The default
depends on the model.
}
Default is \code{linkg = "linear"}, which is equivalent (done internally) to set
all links as \code{"linear"} in the new format. See \link{link.btsr} for valid
links. For details, see the Section \sQuote{The BTSR structure} in
\link{btsr-package}.}

\item{configs.linkg}{a list with two elements, \code{ctt} and \code{power}, which
define the constant \eqn{a} and the exponent \eqn{b} in the link function
\eqn{g(x) = a x^b}. Each element can be specified as a numeric value, a
vector of size 2 or a named list. The default is \code{configs.linkg = NULL}.
See the Section \link[=arguments.link]{Link defaults}  for details.}
}
\value{
For models where \eqn{\nu_t} is not time-varying, returns a list (legacy
format) with starting values for the parameters of the selected model.
Possible outputs are

\item{alpha}{ the intercept.}
\item{beta}{ the coefficients for the regressors.}
\item{phi}{ the AR coefficients.}
\item{theta}{ for BARC models, the parameter associate to the map function.
For any other model, the MA coefficients.}
\item{d}{ the long memory parameter.}
\item{nu}{ distribution related parameter (usually, the precision).}
\item{u0}{ for BARC models, the initial value of the iterated map.}

For models where \eqn{\nu_t} is time-varying, returns a list whose elements
are \code{part1} and \code{part2}. Each element is a list with starting values for the
parameters corresponding to each part o the selected model. Possible outputs
for each part are the same as for the legacy format.
}
\description{
Generates initial values for coefficients in BTSR models.
}
\details{
Parameter initialization is done as follows.

\enumerate{
\item  Legacy flat lists are converted to nested \code{part1}/\code{part2} format.
Link functions and density bounds are validated.

\item \strong{Part 1:} \eqn{\mu_t} related parameters.

A linear model is used to estimate \eqn{\alpha}, \eqn{\boldsymbol
  \beta} and \eqn{\boldsymbol \phi} by setting
\deqn{
  \boldsymbol{Y} = \begin{pmatrix}
  Y_1 \\ \vdots \\ Y_n
  \end{pmatrix}
  \quad \text{and} \quad
  D = \begin{pmatrix}
  1 & X_{11} & \cdots & X_{1s} & g_{12}(Y_0) & \cdots & g_{12}(Y_{1-p}) \\
  \vdots & \vdots & \vdots & \vdots & \vdots & \vdots & \vdots \\
  1 & X_{n1} & \cdots & X_{ns} & g_{12}(Y_{n-1}) & \cdots & g_{12}(Y_{n-p})
  \end{pmatrix}
 }
where \eqn{s} is the number of regressors and \eqn{p} is the AR order, and
solving the linear regression problem \eqn{\boldsymbol{Y} =
 D\boldsymbol{\gamma} + \boldsymbol{\epsilon}} via \code{lm.fit}.

MA coefficients \eqn{\boldsymbol{\theta}} are initialized to zero (though
small non-zero values may help with optimization stability)

The long-memory parameter \eqn{d} starts at 0.01 when estimated

For BARC models:
\itemize{
\item Map parameters use:\cr

\if{html}{\out{<div class="sourceCode">}}\preformatted{   +-------+-------+-------+-------+-------+-------+
   |  map  |   1   |   2   |   3   |   4   |   5   |
   +-------+-------+-------+-------+-------+-------+
   | theta |   3   |  0.5  |  3.5  |  0.5  |  NA   |
   +-------+-------+-------+-------+-------+-------+
}\if{html}{\out{</div>}}

\item \eqn{u_0} defaults to \eqn{\pi/4} when not fixed
}

\item \strong{Part 2:} \eqn{\nu_t} related parameters.

If presented and not time varying, \eqn{\nu} is initialized as follows:
\itemize{
\item \eqn{\nu = 5}, for the Kumaraswamy and the Unit-Weibull distributions,
\item \eqn{
    \nu = \displaystyle\frac{1}{n}\sum_{t=1}^n\dfrac{\hat \mu_t (1 - \hat \mu_t)}{\sigma^2} - 1},
for the Beta distribution,
\item \eqn{\nu = \displaystyle\frac{1}{n}\sum_{t=1}^n\frac{\hat \mu_t^2}{\sigma^2}},
for the Gamma distribution
}
where \eqn{(\hat\mu_1, \cdots, \hat\mu_n)' = D\hat{\boldsymbol{\gamma}}}
are the fitted values from the regression model and \eqn{\sigma^2} is the
estimated variance of the residuals.

If \eqn{\nu} is time varying,
\itemize{
\item set \eqn{\alpha} as \eqn{g_{12}(g_2(\nu))}, with \eqn{\nu}
estimated as in the case where the parameter does not vary on time.

\item set \eqn{\boldsymbol{\beta}},  \eqn{\boldsymbol{\phi}} and
\eqn{\boldsymbol{\theta}} to zero.

\item The long-memory parameter \eqn{d} starts at 0.01 when estimated.
}
}
}
\examples{
mu <- 0.5
nu <- 20

yt <- rbeta(100, shape1 = mu * nu, shape2 = (1 - mu) * nu)
# using the general model BARFIMA
coefs.start(model = "BARFIMA", yt = yt, linkg = "linear")
# same output as the specific model BETA
coefs.start(model = "BETA", yt = yt, linkg = "linear")

yt <- rgamma(100, shape = nu, scale = mu / nu)
coefs.start(model = "GARFIMA", yt = yt, linkg = "linear")

}
