% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions_misc.R
\name{link.btsr}
\alias{link.btsr}
\title{Create a Link for BTSR models}
\usage{
link.btsr(link = "linear")
}
\arguments{
\item{link}{character; one of  \code{"linear"},\code{"polynomial"}, \code{"logit"},
\code{"log"}, \code{"loglog"}, \code{"cloglog"}, \code{"SIP"}. See \sQuote{Details} below.
Default is \code{link = "linear"}.}
}
\value{
An object of class \code{"link-btsr"}, a list with components
\item{linkfun}{Link function \verb{function(mu)} \eqn{g(\mu)}}
\item{linkinv}{Inverse link function \verb{function(eta)} \eqn{g^{-1}(eta)}}
\item{linkdif}{Derivative \verb{function(mu)} \eqn{d\eta/d\mu}}
\item{mu.eta}{Derivative \verb{function(eta)} \eqn{d\mu/d\eta}}
\item{name}{a name to be used for the link}
}
\description{
Given the name of a link, this function returns a link function, an inverse
link function, the derivative   \eqn{d\eta/d\mu} and the derivative
\eqn{d\mu/d\eta}.
}
\details{
The available links are:
\describe{
\item{\code{linear}:}{\eqn{g(x) = ax}, for \eqn{a} real.}

\item{\code{polynomial}:}{\eqn{g(x) = ax^b}, for \eqn{a} and \eqn{b} real.}

\item{\code{logit}:}{\eqn{g(x) = \log((x-l)/(u-x))}}

\item{\code{log}:}{\eqn{g(x) = \log(x - l)}}

\item{\code{loglog}:}{\eqn{g(x) = \log(-\log((x-l)/(u - l)))}}

\item{\code{cloglog}:}{\eqn{g(x) = \log(-\log(1-(x-l)/(u - l)))}}

\item{\code{SIP} (Shifted Inverse Power):}{\eqn{g(x) = 1/(a+x)^b}, with
\eqn{a \in \{0,1\}} and \eqn{b} real.}
}
Here \eqn{l} and \eqn{u} denote the lower and upper bounds of \eqn{x}. The
linear link is a particular case of the polynomial link. It was kept for
compatibility with older versions of the package.

The parameters \eqn{a}, \eqn{b}, \eqn{l}, and \eqn{u} are configured using
the \code{configs.linkg} list, which can include the following elements
\itemize{
\item \code{ctt}: A constant multiplier for the link function (default: 1).
\item \code{power}: The power parameter for polynomial and SIP links (default: 1).
\item \code{lower}: The lower bound for \code{mu} (default: 0).
\item \code{upper}: The upper bound for \code{mu} (default: 1).
}
This list must be  passed to the functions created by \code{link.btsr}, when
invoking them.
}
\examples{
\donttest{
#---------------------------------------------
#  0 < y < 1 and linear link
#---------------------------------------------
mylink <- link.btsr("linear")
y <- 0.8
a <- 3.4
gy <- a * y

# comparing the expected result with the output of the function:
mylink$linkfun(mu = y, configs.linkg = list(ctt = a))
gy

mylink$linkinv(eta = gy, configs.linkg = list(ctt = a))
y

mylink$diflink(mu = y, configs.linkg = list(ctt = a))
a

mylink$mu.eta(eta = gy, configs.linkg = list(ctt = a))
1 / a

# same function, different ctt:
mylink$linkfun(mu = y, configs.linkg = list(ctt = a + 1))

#---------------------------------------------
# For linear link bounds have no effect
#---------------------------------------------
mylink <- link.btsr("linear")
y <- 0.8
a <- 3.4
gy <- a * y

mylink$linkfun(mu = y, configs.linkg = list(ctt = a, lower = 1, upper = 2))
mylink$linkfun(mu = y, configs.linkg = list(ctt = a)) # same result
gy

mylink$linkinv(eta = gy, configs.linkg = list(ctt = a, lower = 1, upper = 2))
y

mylink$diflink(mu = y, configs.linkg = list(ctt = a, lower = 1, upper = 2))
a

mylink$mu.eta(eta = gy, configs.linkg = list(ctt = a, lower = 1, upper = 2))
1 / a

#---------------------------------------------
# 0 < y < 1 and logit link
#---------------------------------------------
mylink <- link.btsr("logit")
y <- 0.8
gy <- log(y / (1 - y))
ginv <- 1 / (1 + exp(-gy))

mylink$linkfun(mu = y)
gy

mylink$linkinv(eta = gy)
ginv

mylink$diflink(mu = y)
1 / (y - y**2)

mylink$mu.eta(eta = gy)
ginv - ginv**2

#---------------------------------------------
# 1 < y < 2 and logit link
#---------------------------------------------
mylink <- link.btsr("logit")
a <- 1
b <- 2
y <- 1.8
gy <- log((y - a) / (b - y))
ginv <- b / (1 + exp(-gy)) + a / (1 + exp(gy))

mylink$linkfun(mu = y, configs.linkg = list(lower = 1, upper = 2))
gy

mylink$linkinv(eta = gy, configs.linkg = list(lower = 1, upper = 2))
ginv

mylink$diflink(mu = y, configs.linkg = list(lower = 1, upper = 2))
(b - a) / ((a + b) * y - y**2 - a * b)

mylink$mu.eta(eta = gy, configs.linkg = list(lower = 1, upper = 2))
((a + b) * ginv - ginv**2 - a * b) / (b - a)
}
}
