% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main.R
\name{bayesian_disaggregate}
\alias{bayesian_disaggregate}
\title{Run Bayesian disaggregation}
\usage{
bayesian_disaggregate(
  path_cpi,
  path_weights,
  method = c("weighted", "multiplicative", "dirichlet", "adaptive"),
  lambda = 0.7,
  gamma = 0.1,
  coh_mult = 3,
  coh_const = 0.5,
  stab_a = 1000,
  stab_b = 10,
  stab_kappa = 50,
  likelihood_pattern = "recent"
)
}
\arguments{
\item{path_cpi}{Path to the CPI Excel file. Must contain at least the
columns \code{date} and \code{value} (case/locale tolerant in \code{read_cpi()}).}

\item{path_weights}{Path to the Excel file with the baseline weight matrix
(prior): either \eqn{T \times K} (years in rows, sectors in columns) or a
length-\eqn{K} vector (constant across time). Rows are renormalized to the simplex.}

\item{method}{Disaggregation method: \code{"weighted"}, \code{"multiplicative"},
\code{"dirichlet"}, or \code{"adaptive"}.}

\item{lambda}{Weight for the \code{"weighted"} method in \eqn{[0,1]}. Ignored otherwise.}

\item{gamma}{Uncertainty factor for the \code{"dirichlet"} method (\eqn{> 0}).}

\item{coh_mult}{Multiplier for the coherence increment \eqn{\Delta\rho}.}

\item{coh_const}{Constant offset for coherence, truncated to \eqn{[0,1]}.}

\item{stab_a}{Sensitivity for row-sum deviation penalty \eqn{| \sum w - 1 |}.}

\item{stab_b}{Sensitivity for negative-values penalty (count of negatives).}

\item{stab_kappa}{Sensitivity for temporal variation (average \eqn{|\Delta|}).}

\item{likelihood_pattern}{Temporal spreading pattern for the likelihood:
\code{"constant"}, \code{"recent"}, \code{"linear"}, or \code{"bell"}.}
}
\value{
A \code{list} with:
\describe{
  \item{\code{years}}{Integer vector of years used.}
  \item{\code{industries}}{Character vector of sector/column names.}
  \item{\code{prior}}{Tibble prior \eqn{T \times (1+K)} with \code{Year} then sectors.}
  \item{\code{likelihood_t}}{Tibble likelihood over time (same shape as \code{prior}).}
  \item{\code{likelihood}}{Tibble \code{Sector}, \code{L} (length \eqn{K}).}
  \item{\code{posterior}}{Tibble posterior \eqn{T \times (1+K)} (rows sum to 1).}
  \item{\code{metrics}}{Tibble with hyperparameters + \code{coherence}, \code{stability},
        \code{interpretability}, \code{efficiency}, \code{composite}, \code{T}, \code{K}.}
}
}
\description{
Performs Bayesian disaggregation of an aggregated time series (e.g., CPI)
into \eqn{K} components using one of four deterministic update rules:
\emph{weighted}, \emph{multiplicative}, \emph{dirichlet}, \emph{adaptive}.
}
\details{
Assumptions: (i) prior/posterior rows lie on the simplex; (ii) no MCMC is used,
updates are analytic/deterministic; (iii) \code{read_*} helpers coerce benign
formatting issues and error on malformed inputs.
}
\examples{
\donttest{
# Minimal synthetic run (no files):
T <- 6; K <- 4
P <- matrix(rep(1/K, T*K), nrow = T)
L <- runif(K); L <- L/sum(L)
LT <- spread_likelihood(L, T, "recent")
W  <- posterior_weighted(P, LT, lambda = 0.7)
}

}
\seealso{
\code{\link{read_cpi}}, \code{\link{read_weights_matrix}},
  \code{\link{compute_L_from_P}}, \code{\link{spread_likelihood}},
  \code{\link{posterior_weighted}}, \code{\link{posterior_multiplicative}},
  \code{\link{posterior_dirichlet}}, \code{\link{posterior_adaptive}},
  \code{\link{coherence_score}}, \code{\link{stability_composite}},
  \code{\link{interpretability_score}}
}
