% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/meffects.R
\name{meffects}
\alias{meffects}
\alias{meffects.cdnet}
\alias{meffects.summary.cdnet}
\alias{meffects.sart}
\alias{meffects.summary.sart}
\title{Marginal Effects for Count Data Models and Tobit Models with Social Interactions}
\usage{
meffects(model, ...)

\method{meffects}{cdnet}(
  model,
  Glist,
  cont.var,
  bin.var,
  type.var,
  Glist.contextual,
  data,
  tol = 1e-10,
  maxit = 500,
  boot = 1000,
  progress = TRUE,
  ncores = 1,
  ...
)

\method{meffects}{summary.cdnet}(
  model,
  Glist,
  cont.var,
  bin.var,
  type.var,
  Glist.contextual,
  data,
  tol = 1e-10,
  maxit = 500,
  boot = 1000,
  progress = TRUE,
  ncores = 1,
  ...
)

\method{meffects}{sart}(
  model,
  Glist,
  cont.var,
  bin.var,
  type.var,
  Glist.contextual,
  data,
  tol = 1e-10,
  maxit = 500,
  boot = 1000,
  progress = TRUE,
  ncores = 1,
  ...
)

\method{meffects}{summary.sart}(
  model,
  Glist,
  cont.var,
  bin.var,
  type.var,
  Glist.contextual,
  data,
  tol = 1e-10,
  maxit = 500,
  boot = 1000,
  progress = TRUE,
  ncores = 1,
  ...
)
}
\arguments{
\item{model}{an object of class \code{cdnet} (\code{summary.cdnet}) or \code{sart} (\code{summary.sart}), output of the function
\code{\link{cdnet}} or \code{\link{sart}}, respectively.}

\item{...}{Additional arguments passed to methods.}

\item{Glist}{The network matrix used to obtain \code{model}. Typically, this is the \code{Glist} argument supplied to
the function \code{\link{cdnet}} or \code{\link{sart}}.}

\item{cont.var}{A character vector of continuous variable names for which the marginal effects should be computed.}

\item{bin.var}{A character vector of binary variable names for which the marginal effects should be computed.}

\item{type.var}{A list indicating "own" and contextual variables that appear in the \code{cont.var} and \code{bin.var} arguments.
The list contains pairs of variable names, with the first element being the "own" variable and the second being the
contextual variable. When a variable has no associated contextual variable, only the variable name is included.
For example, \code{type.var = list(c("x1", "gx1"), c("x2", "gx2"), "x3")} means that \code{gx1} is the contextual variable for \code{x1},
\code{gx2} is the contextual variable for \code{x2}, and \code{x3} has no contextual variable. This information is used to compute the
indirect and total marginal effects for \code{x1}, \code{x2}, and \code{x3}.}

\item{Glist.contextual}{The network matrix used to compute contextual variables, if any are specified in the \code{type.var} argument.
For networks consisting of multiple subnets, \code{Glist} can be a list of subnets, where the \code{m}-th element is an
\code{ns*ns} adjacency matrix, with \code{ns} denoting the number of nodes in the \code{m}-th subnet.}

\item{data}{An optional data frame, list, or environment (or object coercible by \code{\link[base]{as.data.frame}}
to a data frame) containing the variables in the model. If not found in \code{data}, the variables are taken from
\code{environment(model)}, typically the environment from which \code{meffects} is called.}

\item{tol}{The tolerance value used in the fixed-point iteration method to compute \code{y}. The process stops if the
\eqn{\ell_1}-distance between two consecutive values of \code{y} is less than \code{tol}.}

\item{maxit}{The maximum number of iterations in the fixed-point iteration method.}

\item{boot}{The number of bootstrap simulations to compute standard errors and confidence intervals.}

\item{progress}{A logical value indicating whether the progress of the bootstrap simulations should be printed to the console.}

\item{ncores}{Number of CPU cores (threads) used to run the bootstrap process in parallel.}
}
\value{
A list containing:
\describe{
\item{\code{info}}{General information about the model.}
\item{\code{estimate}}{The Maximum Likelihood (ML) estimates of the parameters.}
\item{\code{Ey}}{\eqn{E(y)}, the expected values of the endogenous variable.}
\item{\code{GEy}}{The average of \eqn{E(y)} among peers.}
\item{\code{cov}}{A list containing covariance matrices (if \code{cov = TRUE}).}
\item{\code{details}}{Additional outputs returned by the optimizer.}
\item{\code{meffects}}{A list containing the marginal effects.}
}
}
\description{
\code{meffects} computes marginal effects for count data and Tobit models with social interactions.
It is a generic function which means that new printing methods can be easily added for new classes.
}
\examples{
\donttest{
#' set.seed(123)
M      <- 5 # Number of sub-groups
nvec   <- round(runif(M, 100, 200))
n      <- sum(nvec)

# Adjacency matrix
A      <- list()
for (m in 1:M) {
  nm           <- nvec[m]
  Am           <- matrix(0, nm, nm)
  max_d        <- 30 #maximum number of friends
  for (i in 1:nm) {
    tmp        <- sample((1:nm)[-i], sample(0:max_d, 1))
    Am[i, tmp] <- 1
  }
  A[[m]]       <- Am
}
Anorm  <- norm.network(A) #Row-normalization

# X
X      <- cbind(rnorm(n, 1, 3), rexp(n, 0.4))

# Two group:
group  <- 1*(X[,1] > 0.95)

# Networks
# length(group) = 2 and unique(sort(group)) = c(0, 1)
# The networks must be defined as to capture:
# peer effects of `0` on `0`, peer effects of `1` on `0`
# peer effects of `0` on `1`, and peer effects of `1` on `1`
G        <- list()
cums     <- c(0, cumsum(nvec))
for (m in 1:M) {
  tp     <- group[(cums[m] + 1):(cums[m + 1])]
  Am     <- A[[m]]
  G[[m]] <- norm.network(list(Am * ((1 - tp) \%*\% t(1 - tp)),
                              Am * ((1 - tp) \%*\% t(tp)),
                              Am * (tp \%*\% t(1 - tp)),
                              Am * (tp \%*\% t(tp))))
}

# Parameters
lambda <- c(0.2, 0.3, -0.15, 0.25) 
Gamma  <- c(4.5, 2.2, -0.9, 1.5, -1.2)
delta  <- rep(c(2.6, 1.47, 0.85, 0.7, 0.5), 2) 

# Data
data   <- data.frame(X, peer.avg(Anorm, cbind(x1 = X[,1], x2 =  X[,2])))
colnames(data) = c("x1", "x2", "gx1", "gx2")

ytmp   <- simcdnet(formula = ~ x1 + x2 + gx1 + gx2, Glist = G, Rbar = rep(5, 2),
                   lambda = lambda, Gamma = Gamma, delta = delta, group = group,
                   data = data)
y      <- ytmp$y
hist(y, breaks = max(y) + 1)
table(y)

# Estimation
est    <- cdnet(formula = y ~ x1 + x2 + gx1 + gx2, Glist = G, Rbar = rep(5, 2), group = group,
                optimizer = "fastlbfgs", data = data,
                opt.ctr = list(maxit = 5e3, eps_f = 1e-11, eps_g = 1e-11))

meffects(est, Glist = G, data = data, cont.var = c("x1", "x2", "gx1", "gx2"),
         type.var = list(c("x1", "gx1"), c("x2", "gx2")), Glist.contextual = Anorm,
         boot = 100, ncores = 2)
}
}
