#' Initialize input vectors for the ACMTF algorithm
#'
#' @inheritParams initializeCMTF
#' @param Z List object as generated by [setupCMTFdata()].
#' @param numComponents Integer stating the number of desired components for the CMTF model.
#' @param initialization Initialization method, either "random" or "nvec" (default "random"). Random will initialize random input vectors. Nvec will initialize vectors according to an singular value decomposition of the (matricized, if needed) concatenated datasets per mode.
#' @param output How to return output: as a "Fac" object (default) or vectorized ("vect").
#'
#' @return List or vector of initialized input vectors per mode.
#' @export
#'
#' @examples
#' A = array(rnorm(108*2), c(108, 2))
#' B = array(rnorm(100*2), c(100, 2))
#' C = array(rnorm(10*2), c(10, 2))
#' D = array(rnorm(100*2), c(100,2))
#' E = array(rnorm(10*2), c(10,2))
#'
#' df1 = reinflateTensor(A, B, C)
#' df2 = reinflateTensor(A, D, E)
#' datasets = list(df1, df2)
#' modes = list(c(1,2,3), c(1,4,5))
#' Z = setupCMTFdata(datasets, modes, normalize=FALSE)
#'
#' init = initializeACMTF(Z, 2)
initializeACMTF = function(Z, numComponents, initialization="random", output="Fac", Y=NULL){

  numModes = max(unlist(Z$modes))
  numDatasets = length(Z$object)
  init = initializeCMTF(Z, numComponents, initialization, output="Fac", Y=Y)

  # Normalize the initialized values to norm 1
  newInit = list()
  for(i in 1:numModes){
    newInit[[i]] = array(0L, dim(init[[i]]))
    for(j in 1:numComponents){
      newInit[[i]][,j] = init[[i]][,j] / rTensor::fnorm(rTensor::as.tensor(init[[i]][,j]))
    }
  }
  init = newInit

  # Initialize lambdas values as 1
  init[[numModes+1]] = array(1, c(numDatasets,numComponents))

  if(output=="vect"){
    return(fac_to_vect(init))
  }
  else{
    return(init)
  }
}
