% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ships_methods.R
\name{make.construction.traffic}
\alias{make.construction.traffic}
\title{Generate artificial AIS data representing ship traffic during wind farm construction}
\usage{
make.construction.traffic(
  pilings,
  ships = NULL,
  x.harbour,
  y.harbour,
  startday = "2010-01-01",
  tz = "UTC"
)
}
\arguments{
\item{pilings}{A data frame containing the positions and times of piling operations during the construction of a wind farm.
May contain real data but must be in the format as produced by \code{\link{make.windfarms}}: columns 'id', 'x.coordinate' (num),
'y.coordinate' (num),    'impact' (num; optional - not required for this function),    'tick.start' (num), and    'tick.end' (num).}

\item{ships}{A data frame with the characteristics of the ships that should be simulated. Must contain one entry for each ship and
the columns 'id', length' (num; ship length, meters), 'speed' (num; knots), and 'daily.pause' (num; length of active, i.e. noisy,
pause at each piling event, ticks). If no data are provided, a set of 13 ships based on data from piling operations at the Moray East wind farm in 2019 is used (see details).}

\item{x.harbour}{Numeric. X coordinates (meters) of originating harbour for all ships}

\item{y.harbour}{Numeric. Y coordinates (meters) of originating harbour for all ships}

\item{startday}{Character. Intended start date of the simulation. If ticks provided in 'pilings' were converted from real dates using \code{\link{time.to.tick}},
this should be the same as the 'origin' used in that conversion. Defaults to "2010-01-01".}

\item{tz}{Time zone. Defaults to "UTC}
}
\value{
A data frame of ship positions and the times at which the positions are set, with columns 'id', 'time' (of the form
"%Y-%m-%d %H:%M:%S", character, see \code{\link{as.POSIXct}}), 'type' (char; ship' type, set to "Other"), 'length' (num; ship length, meters),
'x', and 'y' (num; position, meters). This data frame is suitable for conversion using \code{\link{ais.to.DeponsShips}}.
}
\description{
Generates artificial Automatic Identification System (AIS) data to represent ship traffic connected with the construction
of a wind farm, to supplement scenarios where no real records of such data are available. A list of ship characteristics may be
supplied, or a built-in default set of ships may be used. Ship routes are constructed such that all ships start at a chosen harbour position,
attend each piling event observing individual stay durations, and between pilings either return to the harbour or traverse as much of the
route back to the harbour as time allows. Route data can afterwards be converted to a \code{DeponsShips} object (using
\code{\link{ais.to.DeponsShips}}) which can be read by DEPONS.
}
\details{
All ships will attend each piling event. The route of each ship over the wind farm construction period is created in this manner:
1) The ship will try to be in position at a piling for the duration of its 'pause.length'; this is considered an active pause, i.e. the ship
holds position by engine use and generates noise. The pause is timed such that the piling event's midpoint is in the middle of the pause.
2) The ship starts at the harbour location at the beginning of the data set, and will leave for the first piling in time to reach it at
its given speed.
3) Between any two piling events, the ship will attempt to go back to the harbour if there is sufficient time to do so, and wait there until
the next event (see ship ID_1 in example). If there is insufficient time to cover the whole distance back and forth, the ship will go back as far as possible along the
route, and turn around in time to reach the next piling event (see ship ID_2 in example). If the time between pilings is shorter than the active pause duration, the ship
will move straight between piling events, spending half of the remaining time pausing at each location. (When assigning active pauses later using \code{\link{make.stationary.ships}},
pausing at the harbour will not be considered an active pause, i.e. no noise will be generated.)
4) After the last piling event, the ship will return to the harbour and wait there until the end of the simulation.

If the number of ticks until the first piling event is too low for the slowest ship in the set to reach it from the harbour,
an error is thrown, in which case the time of all piling events should be shifted backward. If the ticks were converted from real dates using
\code{\link{time.to.tick}}, this can be done by choosing an earlier 'origin' in that conversion.

Ships can only move in a straight line between the harbour and any piling event. The harbour position should therefore be chosen so that such
routes do not cut across islands, headlands etc. (land intersections will however not interfere with movement, and the user may consider ignoring
minor irregularities for the sake of convenience).

Function \code{\link{make.windfarms}} generates hypothetical piling data that can be used with this function. If real piling data are used,
dates should be converted to ticks using \code{\link{time.to.tick}}.

The type of all ships is set to "Other", since this category includes those vessel classes expected to be present at piling operations.

After the data have been generated, the user's next step will probably be conversion to a \code{DeponsShips} object with \code{\link{ais.to.DeponsShips}},
and parameterization of active pauses using \code{\link{make.stationary.ships}}. It is recommended to also use \code{\link{check.DeponsShips}} to
verify ship speeds.

If no ship data are provided, a set of 13 ships based on data from piling operations at the Moray East wind farm in 2019 is used:
\tabular{ccc}{
 \strong{vessel class} \tab \strong{length (m)} \tab \strong{active pause duration (ticks)} \cr
CTV \tab 14 \tab 0\cr
CTV \tab 15 \tab 1\cr
CTV \tab 19 \tab 8\cr
CTV \tab 23 \tab 1\cr
CTV \tab 24 \tab 1\cr
CTV \tab 25 \tab 4\cr
CTV \tab 25.75 \tab 1\cr
CTV \tab 25.75 \tab 2\cr
CTV \tab 27 \tab 1\cr
Dive \tab 20 \tab 1\cr
OS \tab 88.4 \tab 11\cr
OS \tab 89 \tab 41\cr
OS \tab 120 \tab 9
}
(CTV: crew transfer vessel, Dive: dive support ship, OS: offshore supply ship)

The type of these ships is set to "Other" and the speed to 7.4 knots (see \code{\link{make.stationary.ships}} for the source of these conventions).

\strong{Ships remaining at sea for the duration of the complete piling operations}

One or more principal construction ships may be intended to remain within the piling area for the entire duration of piling operations without returning to harbour.
This is achieved by setting the ship's 'daily.pause' duration in the 'ships' dataframe to a high number of ticks (greater than the longest gap between pilings,
e.g. multiple days of 48 ticks each), which will cause the ship to progress directly from one piling to the next. This value will be capped at 48 ticks before the first 
and after the last piling event. Note that all this time spent pausing at sea will be parameterized as an active (noisy) pause when the generated ship data set is later 
processed with \code{\link{make.stationary.ships}}. This may be correct for crane ships or similar that actively hold position at sea, but inappropriate for jack-up vessels
that take up a fixed position at the piling. In the latter case, the user should make note of the ship's identifier, and after carrying out the 'check' step of processing 
with 'make.stationary.ships', remove all of the ship's entries from the candidates data frame before carrying out the 'replace' step (see \code{\link{make.stationary.ships}} for details).
}
\examples{
x.harbour <- 0
y.harbour <- 0
ships <- as.data.frame(rbind(c("ID_1", 20, 14, 2),
                              c("ID_2", 20, 8, 20)))
ships[,2:4] <- as.numeric(unlist(ships[,2:4]))
colnames(ships) <- c("id", "length", "speed", "pause.length")
pilings <- as.data.frame(rbind(c("Piling_1", 100000, 100000, 50, 54),
                               c("Piling_2", 102000, 100000, 80, 84),
                               c("Piling_3", 100000, 102000, 110, 114),
                               c("Piling_4", 102000, 102000, 140, 144)))
pilings[,2:5] <- as.numeric(unlist(pilings[,2:5]))
colnames(pilings) <- c("id", "x.coordinate", "y.coordinate", "tick.start", "tick.end")
construction.traffic <- make.construction.traffic(pilings = pilings, ships = ships,
                                       x.harbour = x.harbour, y.harbour = y.harbour)
}
\seealso{
\code{\link{make.windfarms}} for creation of hypothetical wind farm piling data.
}
