% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DPI.R
\name{DPI}
\alias{DPI}
\title{The Directed Prediction Index (DPI).}
\usage{
DPI(
  model,
  x,
  y,
  data = NULL,
  k.cov = 1,
  n.sim = 1000,
  alpha = 0.05,
  bonf = FALSE,
  pseudoBF = FALSE,
  seed = NULL,
  progress,
  file = NULL,
  width = 6,
  height = 4,
  dpi = 500
)
}
\arguments{
\item{model}{Model object (\code{lm}).}

\item{x}{Independent (predictor) variable.}

\item{y}{Dependent (outcome) variable.}

\item{data}{[Optional] Defaults to \code{NULL}. If \code{data} is specified, then \code{model} will be ignored and a linear model \code{lm({y} ~ {x} + .)} will be fitted inside. This is helpful for exploring all variables in a dataset.}

\item{k.cov}{Number of random covariates (simulating potential omitted variables) added to each simulation sample.
\itemize{
\item Defaults to \code{1}. Please also test different \code{k.cov} values as robustness checks (see \code{\link[=DPI_curve]{DPI_curve()}}).
\item If \code{k.cov} > 0, the raw data (without bootstrapping) are used, with \code{k.cov} random variables appended, for simulation.
\item If \code{k.cov} = 0 (not suggested), bootstrap samples (resampling with replacement) are used for simulation.
}}

\item{n.sim}{Number of simulation samples. Defaults to \code{1000}.}

\item{alpha}{Significance level for computing the \code{Significance} score (0~1) based on \emph{p} value of partial correlation between \code{X} and \code{Y}. Defaults to \code{0.05}.
\itemize{
\item \code{Direction = R2.Y - R2.X}
\item \code{Significance = 1 - tanh(p.beta.xy/alpha/2)}
}}

\item{bonf}{Bonferroni correction to control for false positive rates: \code{alpha} is divided by, and \emph{p} values are multiplied by, the number of comparisons.
\itemize{
\item Defaults to \code{FALSE}: No correction, suitable if you plan to test only one pair of variables.
\item \code{TRUE}: Using \code{k * (k - 1) / 2} (all pairs of variables) where \code{k = length(data)}.
\item A user-specified number of comparisons.
}}

\item{pseudoBF}{Use normalized pseudo Bayes Factors \code{sigmoid(log(PseudoBF10))} alternatively as the \code{Significance} score (0~1). Pseudo Bayes Factors are computed from \emph{p} value of X-Y partial relationship and total sample size, using the transformation rules proposed by Wagenmakers (2022) \doi{10.31234/osf.io/egydq}.

Defaults to \code{FALSE} because it makes less penalties for insignificant partial relationships between \code{X} and \code{Y}, see Examples in \code{\link[=DPI]{DPI()}} and \href{https://psychbruce.github.io/DPI/#step-2-normalized-penalty-as-significance-score}{online documentation}.}

\item{seed}{Random seed for replicable results. Defaults to \code{NULL}.}

\item{progress}{Show progress bar. Defaults to \code{FALSE} (if \code{n.sim} < 5000).}

\item{file}{File name of saved plot (\code{".png"} or \code{".pdf"}).}

\item{width, height}{Width and height (in inches) of saved plot. Defaults to \code{6} and \code{4}.}

\item{dpi}{Dots per inch (figure resolution). Defaults to \code{500}.}
}
\value{
Return a data.frame of simulation results:
\itemize{
\item \code{DPI = Direction * Significance}
\itemize{
\item \verb{= (R2.Y - R2.X) * (1 - tanh(p.beta.xy/alpha/2))}
\itemize{
\item if \code{pseudoBF=FALSE} (default, suggested)
\item more conservative estimates
}
\item \verb{= (R2.Y - R2.X) * plogis(log(pseudo.BF.xy))}
\itemize{
\item if \code{pseudoBF=TRUE}
\item less conservative for insignificant X-Y relationship
}
}
\item \code{delta.R2}
\itemize{
\item \code{R2.Y - R2.X}
}
\item \code{R2.Y}
\itemize{
\item \eqn{R^2} of regression model predicting Y using X and all other covariates
}
\item \code{R2.X}
\itemize{
\item \eqn{R^2} of regression model predicting X using Y and all other covariates
}
\item \code{t.beta.xy}
\itemize{
\item \emph{t} value for coefficient of X predicting Y (always equal to \emph{t} value for coefficient of Y predicting X) when controlling for all other covariates
}
\item \code{p.beta.xy}
\itemize{
\item \emph{p} value for coefficient of X predicting Y (always equal to \emph{p} value for coefficient of Y predicting X) when controlling for all other covariates
}
\item \code{df.beta.xy}
\itemize{
\item residual degree of freedom (df) of \code{t.beta.xy}
}
\item \code{r.partial.xy}
\itemize{
\item partial correlation (always with the same \emph{t} value as \code{t.beta.xy}) between X and Y when controlling for all other covariates
}
\item \code{sigmoid.p.xy}
\itemize{
\item sigmoid \emph{p} value as \code{1 - tanh(p.beta.xy/alpha/2)}
}
\item \code{pseudo.BF.xy}
\itemize{
\item pseudo Bayes Factors (\eqn{BF_{10}}) computed from \emph{p} value \code{p.beta.xy} and sample size \code{nobs(model)}, see \code{\link[=p_to_bf]{p_to_bf()}}
}
}
}
\description{
The Directed Prediction Index (DPI) is a quasi-causal inference method for cross-sectional data designed to quantify the \emph{relative endogeneity} (relative dependence) of outcome (\emph{Y}) vs. predictor (\emph{X}) variables in regression models. By comparing the proportion of variance explained (\emph{R}-squared) between the \emph{Y}-as-outcome model and the \emph{X}-as-outcome model while controlling for a sufficient number of possible confounders, it can suggest a plausible (admissible) direction of influence from a more exogenous variable (\emph{X}) to a more endogenous variable (\emph{Y}). Methodological details are provided at \url{https://psychbruce.github.io/DPI/}.
}
\examples{
\donttest{# input a fitted model
model = lm(Ozone ~ ., data=airquality)
DPI(model, x="Solar.R", y="Ozone", seed=1)  # DPI > 0
DPI(model, x="Wind", y="Ozone", seed=1)     # DPI > 0
DPI(model, x="Solar.R", y="Wind", seed=1)   # unrelated

# or input raw data, test with more random covs
DPI(data=airquality, x="Solar.R", y="Ozone",
    k.cov=10, seed=1)
DPI(data=airquality, x="Wind", y="Ozone",
    k.cov=10, seed=1)
DPI(data=airquality, x="Solar.R", y="Wind",
    k.cov=10, seed=1)

# or use pseudo Bayes Factors for the significance score
# (less conservative for insignificant X-Y relationship)
DPI(data=airquality, x="Solar.R", y="Ozone", k.cov=10,
    pseudoBF=TRUE, seed=1)  # DPI > 0 (true positive)
DPI(data=airquality, x="Wind", y="Ozone", k.cov=10,
    pseudoBF=TRUE, seed=1)  # DPI > 0 (true positive)
DPI(data=airquality, x="Solar.R", y="Wind", k.cov=10,
    pseudoBF=TRUE, seed=1)  # DPI > 0 (false positive!)
}
}
\seealso{
\link{S3method.dpi}

\code{\link[=DPI_curve]{DPI_curve()}}

\code{\link[=DPI_dag]{DPI_dag()}}

\code{\link[=BNs_dag]{BNs_dag()}}

\code{\link[=cor_net]{cor_net()}}

\code{\link[=p_to_bf]{p_to_bf()}}
}
