% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DHolm_fun.R
\name{DHolm}
\alias{DHolm}
\alias{DHolm.default}
\alias{DHolm.DiscreteTestResults}
\title{Discrete Holm Procedure}
\usage{
DHolm(test_results, ...)

\method{DHolm}{default}(
  test_results,
  pCDFlist,
  alpha = 0.05,
  critical_values = FALSE,
  select_threshold = 1,
  pCDFlist_indices = NULL,
  ...
)

\method{DHolm}{DiscreteTestResults}(
  test_results,
  alpha = 0.05,
  critical_values = FALSE,
  select_threshold = 1,
  ...
)
}
\arguments{
\item{test_results}{either a numeric vector with \eqn{p}-values or an R6 object of class \code{\link[DiscreteTests]{DiscreteTestResults}} from package \link[DiscreteTests]{DiscreteTests} for which a discrete FWER procedure is to be performed.}

\item{...}{further arguments to be passed to or from other methods. They are ignored here.}

\item{pCDFlist}{list of the supports of the CDFs of the \eqn{p}-values; each list item must be a numeric vector, which is sorted in increasing order and whose last element equals 1.}

\item{alpha}{single real number strictly between 0 and 1 indicating the target FWER level.}

\item{critical_values}{single boolean specifying whether critical constants are to be computed.}

\item{select_threshold}{single real number strictly between 0 and 1 indicating the largest raw \eqn{p}-value to be considered, i.e. only \eqn{p}-values below this threshold are considered and the procedures are adjusted in order to take this selection effect into account; if \code{select_threshold = 1} (the default), all raw \eqn{p}-values are selected.}

\item{pCDFlist_indices}{list of numeric vectors containing the test indices that indicate to which raw \eqn{p}-value(s) each support in \code{pCDFlist} belongs; if \code{NULL} (the default) the lengths of \code{test_results} and \code{pCDFlist} \strong{must} be equal.}
}
\value{
A \code{DiscreteFWER} S3 class object whose elements are:
\item{Rejected}{rejected raw \eqn{p}-values.}
\item{Indices}{indices of rejected hypotheses.}
\item{Num_rejected}{number of rejections.}
\item{Adjusted}{adjusted \eqn{p}-values.}
\item{Critical_constants}{critical values (only exists if computations where performed with \code{critical_values = TRUE}).}
\item{Data}{list with input data.}
\item{Data$Method}{character string describing the performed algorithm, e.g. 'Discrete Bonferroni procedure'.}
\item{Data$Raw_pvalues}{observed \eqn{p}-values.}
\item{Data$pCDFlist}{list of the \eqn{p}-value supports.}
\item{Data$FWER_level}{FWER level \code{alpha}.}
\item{Data$Independence}{boolean indicating whether the \eqn{p}-values were considered as independent.}
\item{Data$Single_step}{boolean indicating whether a single-step or step-down procedure was performed.}
\item{Data$Data_name}{the respective variable names of the input data.}
\item{Select}{list with data related to \eqn{p}-value selection; only exists if \code{select_threshold < 1}.}
\item{Select$Threshold}{\eqn{p}-value selection threshold (\code{select_threshold}).}
\item{Select$Effective_Thresholds}{results of each \eqn{p}-value CDF evaluated at the selection threshold.}
\item{Select$Pvalues}{selected \eqn{p}-values that are \eqn{\leq} selection threshold.}
\item{Select$Indices}{indices of \eqn{p}-values \eqn{\leq} selection threshold.}
\item{Select$Scaled}{scaled selected \eqn{p}-values.}
\item{Select$Number}{number of selected \eqn{p}-values \eqn{\leq} selection threshold.}
}
\description{
\code{DHolm()} is a wrapper function of \code{\link[=discrete_FWER]{discrete_FWER()}} for computing the
discrete Holm step-down procedure for tests with an arbitrary dependency
structure. It simply passes its arguments to \code{\link[=discrete_FWER]{discrete_FWER()}} with fixed
\code{independence = FALSE} and \code{single_step = FALSE}.
}
\details{
Computing critical constants (\code{critical_values = TRUE}) requires considerably
more execution time, especially if the number of unique supports is large.
We recommend that users should only have them calculated when they need them,
e.g. for illustrating the rejection set in a plot or other theoretical
reasons. Setting (\code{critical_values = FALSE}) is sufficient for obtaining
rejection decisions and adjusted \eqn{p}-values.
}
\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1 - X1
Y2 <- N2 - X2
df <- data.frame(X1, Y1, X2, Y2)
df

# Computation of p-values and their supports with Fisher's exact test
library(DiscreteTests)  # for Fisher's exact test
test_results <- fisher_test_pv(df)
raw_pvalues <- test_results$get_pvalues()
pCDFlist <- test_results$get_pvalue_supports()

# d-Holm without critical values; using extracted p-values and supports
DHolm_fast <- DHolm(raw_pvalues, pCDFlist)
summary(DHolm_fast)

# d-Holm with critical values; using test results object
DHolm_crit <- DHolm(test_results, critical_values = TRUE)
summary(DHolm_crit)

}
\references{
Döhler, S. (2010). Validation of credit default probabilities using
multiple-testing procedures. \emph{Journal of Risk Model Validation}, \emph{4}(4),
59-92. \doi{10.21314/JRMV.2010.062}

Zhu, Y., & Guo, W. (2019). Family-Wise Error Rate Controlling Procedures for
Discrete Data. \emph{Statistics in Biopharmaceutical Research}, \emph{12}(1),
117-128. \doi{10.1080/19466315.2019.1654912}
}
\seealso{
\code{\link[=discrete_FWER]{discrete_FWER()}}, \code{\link[=DBonferroni]{DBonferroni()}}, \code{\link[=DSidak]{DSidak()}}, \code{\link[=DHochberg]{DHochberg()}}
}
