% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/discreteFWER_fun.R
\name{discrete_FWER}
\alias{discrete_FWER}
\alias{discrete_FWER.default}
\alias{discrete_FWER.DiscreteTestResults}
\title{Discrete Family-wise Error Rate (FWER) Adaptation Procedures}
\usage{
discrete_FWER(test_results, ...)

\method{discrete_FWER}{default}(
  test_results,
  pCDFlist,
  alpha = 0.05,
  independence = FALSE,
  single_step = FALSE,
  critical_values = FALSE,
  select_threshold = 1,
  pCDFlist_indices = NULL,
  ...
)

\method{discrete_FWER}{DiscreteTestResults}(
  test_results,
  alpha = 0.05,
  independence = FALSE,
  single_step = FALSE,
  critical_values = FALSE,
  select_threshold = 1,
  ...
)
}
\arguments{
\item{test_results}{either a numeric vector with \eqn{p}-values or an R6 object of class \code{\link[DiscreteTests]{DiscreteTestResults}} from package \link[DiscreteTests]{DiscreteTests} for which a discrete FWER procedure is to be performed.}

\item{...}{further arguments to be passed to or from other methods. They are ignored here.}

\item{pCDFlist}{list of the supports of the CDFs of the \eqn{p}-values; each list item must be a numeric vector, which is sorted in increasing order and whose last element equals 1.}

\item{alpha}{single real number strictly between 0 and 1 indicating the target FWER level.}

\item{independence}{single boolean specifying whether the \eqn{p}-values are statistically independent or not.}

\item{single_step}{single boolean specifying whether to perform a single-step (\code{TRUE}) or step-down (\code{FALSE}; the default) procedure.}

\item{critical_values}{single boolean specifying whether critical constants are to be computed.}

\item{select_threshold}{single real number strictly between 0 and 1 indicating the largest raw \eqn{p}-value to be considered, i.e. only \eqn{p}-values below this threshold are considered and the procedures are adjusted in order to take this selection effect into account; if \code{select_threshold = 1} (the default), all raw \eqn{p}-values are selected.}

\item{pCDFlist_indices}{list of numeric vectors containing the test indices that indicate to which raw \eqn{p}-value(s) each support in \code{pCDFlist} belongs; if \code{NULL} (the default) the lengths of \code{test_results} and \code{pCDFlist} \strong{must} be equal.}
}
\value{
A \code{DiscreteFWER} S3 class object whose elements are:
\item{Rejected}{rejected raw \eqn{p}-values.}
\item{Indices}{indices of rejected hypotheses.}
\item{Num_rejected}{number of rejections.}
\item{Adjusted}{adjusted \eqn{p}-values.}
\item{Critical_constants}{critical values (only exists if computations where performed with \code{critical_values = TRUE}).}
\item{Data}{list with input data.}
\item{Data$Method}{character string describing the performed algorithm, e.g. 'Discrete Bonferroni procedure'.}
\item{Data$Raw_pvalues}{observed \eqn{p}-values.}
\item{Data$pCDFlist}{list of the \eqn{p}-value supports.}
\item{Data$FWER_level}{FWER level \code{alpha}.}
\item{Data$Independence}{boolean indicating whether the \eqn{p}-values were considered as independent.}
\item{Data$Single_step}{boolean indicating whether a single-step or step-down procedure was performed.}
\item{Data$Data_name}{the respective variable names of the input data.}
\item{Select}{list with data related to \eqn{p}-value selection; only exists if \code{select_threshold < 1}.}
\item{Select$Threshold}{\eqn{p}-value selection threshold (\code{select_threshold}).}
\item{Select$Effective_Thresholds}{results of each \eqn{p}-value CDF evaluated at the selection threshold.}
\item{Select$Pvalues}{selected \eqn{p}-values that are \eqn{\leq} selection threshold.}
\item{Select$Indices}{indices of \eqn{p}-values \eqn{\leq} selection threshold.}
\item{Select$Scaled}{scaled selected \eqn{p}-values.}
\item{Select$Number}{number of selected \eqn{p}-values \eqn{\leq} selection threshold.}
}
\description{
Apply a discrete FWER adaptation procedure, with or without computing the
critical values, to a set of p-values and their discrete support.
}
\details{
Computing critical constants (\code{critical_values = TRUE}) requires considerably
more execution time, especially if the number of unique supports is large.
We recommend that users should only have them calculated when they need them,
e.g. for illustrating the rejection set in a plot or other theoretical
reasons. Setting (\code{critical_values = FALSE}) is sufficient for obtaining
rejection decisions and adjusted \eqn{p}-values.

Depending on the choices of \code{independence} and \code{single_step}, one of the
following procedures, is applied:\tabular{lcc}{
    \tab single-step \tab stepwise \cr
   independent \tab Šidák \tab Hochberg (step-up) \cr
   not independent \tab Bonferroni \tab Holm (step-down) \cr
}


Each procedure is available by its own shortcut function:\tabular{lcc}{
    \tab single-step \tab stepwise \cr
   independent \tab \code{DSidak()} \tab \code{DHochberg()} \cr
   not independent \tab \code{DBonferroni()} \tab \code{DHolm()} \cr
}
}
\examples{
X1 <- c(4, 2, 2, 14, 6, 9, 4, 0, 1)
X2 <- c(0, 0, 1, 3, 2, 1, 2, 2, 2)
N1 <- rep(148, 9)
N2 <- rep(132, 9)
Y1 <- N1 - X1
Y2 <- N2 - X2
df <- data.frame(X1, Y1, X2, Y2)
df

# Computation of p-values and their supports with Fisher's exact test
library(DiscreteTests)  # for Fisher's exact test
test_results <- fisher_test_pv(df)
raw_pvalues <- test_results$get_pvalues()
pCDFlist <- test_results$get_pvalue_supports()

# d-Holm without critical values; using test results object
DFWER_dep_sd_fast <- discrete_FWER(test_results)
summary(DFWER_dep_sd_fast)

# d-Holm with critical values; using extracted p-values and supports
DFWER_dep_sd_crit <- discrete_FWER(raw_pvalues, pCDFlist, 
                                   critical_values = TRUE)
summary(DFWER_dep_sd_crit)

# d-Bonferroni without critical values; using test results object
DFWER_dep_fast <- discrete_FWER(test_results, single_step = TRUE)
summary(DFWER_dep_fast)

# d-Bonferroni with critical values; using extracted p-values and supports
DFWER_dep_crit <- discrete_FWER(raw_pvalues, pCDFlist, single_step = TRUE,
                                critical_values = TRUE)
summary(DFWER_dep_crit)

# d-Hochberg without critical values; using test results object
DFWER_ind_su_fast <- discrete_FWER(test_results, independence = TRUE)
summary(DFWER_ind_su_fast)

# d-Hochberg with critical values; using extracted p-values and supports
DFWER_ind_su_crit <- discrete_FWER(raw_pvalues, pCDFlist, 
                                   independence = TRUE,
                                   critical_values = TRUE)
summary(DFWER_ind_su_crit)

# d-Šidák without critical values; using extracted p-values and supports
DFWER_ind_fast <- discrete_FWER(raw_pvalues, pCDFlist,
                                independence = TRUE,
                                single_step = TRUE)
summary(DFWER_ind_fast)

# d-Šidák with critical values; using test results object
DFWER_ind_crit <- discrete_FWER(test_results, independence = TRUE,
                                single_step = TRUE, 
                                critical_values = TRUE)
summary(DFWER_ind_crit)

}
\references{
Döhler, S. (2010). Validation of credit default probabilities using
multiple-testing procedures. \emph{Journal of Risk Model Validation}, \emph{4}(4),
59-92. \doi{10.21314/JRMV.2010.062}

Zhu, Y., & Guo, W. (2019). Family-Wise Error Rate Controlling Procedures for
Discrete Data. \emph{Statistics in Biopharmaceutical Research}, \emph{12}(1),
117-128. \doi{10.1080/19466315.2019.1654912}
}
\seealso{
\code{\link[=DiscreteFWER-package]{DiscreteFWER}}, \code{\link[=DBonferroni]{DBonferroni()}}, \code{\link[=DHolm]{DHolm()}},
\code{\link[=DSidak]{DSidak()}}, \code{\link[=DHochberg]{DHochberg()}}
}
