% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/effect.stars.R
\name{effectstars.default}
\alias{effectstars.default}
\alias{effectstars}
\title{Plot effect stars.}
\usage{
\method{effectstars}{default}(x, names = NULL, subs = NULL,
  labels = NULL, control = star.ctrl(), cols = NULL, fixed = FALSE,
  scale = 1, ...)
}
\arguments{
\item{x}{A matrix containing all coefficients to plot, one column per group/covariate, 
one row per category. If the arguments \code{names} and \code{labels} are not specified, 
the \code{colnames} and \code{rownames} of \code{x} are used.}

\item{names}{A vector containing all group/covariate names, will be used as titles of single 
effect stars. If NULL, \code{colnames} of \code{x} is used.}

\item{subs}{A vector containing all subtitles, one per group/covariate.}

\item{labels}{A vector or a matrix containing labels of the categories. If labels is a matrix, 
it needs to have the same dimensions as \code{x}. Otherwise, \code{labels} is a vector with 
length equal to the number of categories, i.e. number rows of \code{x}. If NULL, \code{rownames}
of \code{x} is used.}

\item{control}{Control argument (to set graphical parameters) for method \code{\link{effectstars}}, see \code{\link{star.ctrl}}.}

\item{cols}{Number of columns for arranging effect stars}

\item{fixed}{If \code{TRUE}, all circles have the same radius. If \code{FALSE}, every star is scaled so that 
the length of the longest ray is equal for all stars.}

\item{scale}{Global factor to increase (\code{scale}>1) or decrease (\code{scale}<1) the size of the stars.}

\item{...}{possible further arguments}
}
\description{
Plots effect stars for grouped coefficients. Effect stars are applicable if the
parameters of a model are grouped in some sense. For example, the parameters of a multinomial 
logit models are grouped by the covariates, i.e. per covariate there is one estimate per response category.
But also in many other models, the parameters can have a fixed grouping structure. All estimates have to be positive, 
typically the exponentials of the estimates are plotted. Every effect star comes with a circle of radius 1.
This circle represents the case of no effect, i.e. \code{exp(0)=1}.
}
\examples{
\dontrun{
#####################
### Simple example for basic effectstars function
p <- 4; k <- 5
coefs <- matrix(exp(rnorm(p*k,sd=0.5)),ncol=k)
rownames(coefs) <- paste("Variable",1:p)
colnames(coefs) <- paste("Cat",1:k)
effectstars(coefs)

#####################
### Example for effect stars for a multivariate logit model
data(xs.nz, package = "VGAMdata")
xs.nz$age <- scale(xs.nz$age)
library(VGAM)

cats_dogs <- vglm(cbind(cat, dog) ~ age +  sex + marital, 
                  data = xs.nz, family = binom2.or(zero = NULL))

summary(cats_dogs)

## quick and dirty
effectstars(exp(coef(cats_dogs, matrix = TRUE)))


## make it pretty
# create the effects matrix you want to plot, name rows and columns
effects <- exp(coef(cats_dogs, matrix = TRUE))
colnames(effects) <- c("cat", "dog", "OR")
rownames(effects) <- c("Intercept", "Age", "Gender", rep("Marital", 3))

# create subtitles containing category labels of predictors
subs <- c(rep("",2), "(male)", "(married)", "(separated/divorced)", "(widowed)")

# create labels containing the response categories and all p-values
p_values <- formatC(summary(cats_dogs)@coef3[,4], format="f", digits=3)
labels <- matrix(paste0(rep(c("cat", "dog", "OR"), nrow(effects)), "\\n(", p_values, ")"),
byrow = TRUE, ncol = 3)

# plot effectstars
effectstars(effects, labels = labels, subs = subs)


#####################
## Example for method effectstars.vglm for a multinomial logit model calculated in VGAM
data(election) 
library(VGAM)
m_elect <- vglm(Partychoice ~ Gender + West + Age + Union + Highschool + Unemployment
+ Pol.Interest + Democracy + Religion, family = multinomial(), data = election)
effectstars(m_elect)
}
}
\references{
Tutz, G. and Schauberger, G. (2013): \emph{Visualization of Categorical Response Models -
from Data Glyphs to Parameter Glyphs}, Journal of Computational and Graphical Statistics 22(1), 156--177.
\cr \cr  Gerhard Tutz (2012): \emph{Regression for Categorical Data}, Cambridge University Press
}
\seealso{
\code{\link{star.ctrl}}, \code{\link{effectstars.vglm}}, \code{\link{effectstars.DIFlasso}}, \code{\link{effectstars.DIFboost}}
}
\author{
Gunther Schauberger \cr
\email{gunther.schauberger@tum.de} \cr \cr
\url{https://www.sg.tum.de/epidemiologie/team/schauberger/}
}
