% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EllDistrEst-adaptive.R
\name{EllDistrEst.adapt}
\alias{EllDistrEst.adapt}
\title{Estimation of the generator of the elliptical distribution by kernel smoothing
with adaptive choice of the bandwidth}
\usage{
EllDistrEst.adapt(
  X,
  mu = 0,
  Sigma_m1 = diag(NCOL(X)),
  grid,
  h_firstStep,
  grid_a = NULL,
  Kernel = "gaussian",
  mpfr = FALSE,
  precBits = 100,
  dopb = TRUE
)
}
\arguments{
\item{X}{a matrix of size \eqn{n \times d}, assumed to be \eqn{n} i.i.d. observations
(rows) of a \eqn{d}-dimensional elliptical distribution.}

\item{mu}{mean of X. This can be the true value or an estimate. It must be
a vector of dimension \eqn{d}.}

\item{Sigma_m1}{inverse of the covariance matrix of X.
This can be the true value or an estimate. It must be
a matrix of dimension \eqn{d \times d}.}

\item{grid}{vector containing the values at which we want the generator to be
estimated.}

\item{h_firstStep}{a vector of size \code{2} containing first-step bandwidths
to be used. The first one is used for the estimation of the asymptotic mean-squared
error. The second one is used for the first step estimation of \eqn{g}.
From these two estimators, a final value of the bandwidth \eqn{h} is determined,
which is used for the final estimator of \eqn{g}.

If \code{h_firstStep} is of length \code{1}, its value is reused for both purposes
(estimation of the AMSE and first-step estimation of \eqn{g}).}

\item{grid_a}{the grid of possible values of \code{a} to be used.
If missing, a default sequence is used.}

\item{Kernel}{name of the kernel. Possible choices are
\code{"gaussian"}, \code{"epanechnikov"}, \code{"triangular"}.}

\item{mpfr}{if \code{mpfr = TRUE}, multiple precision floating point is used
via the package \link[Rmpfr:Rmpfr]{Rmpfr}.
This allows for a higher (numerical) accuracy, at the expense of computing time.
It is recommended to use this option for higher dimensions.}

\item{precBits}{number of precBits used for floating point precision
(only used if \code{mpfr = TRUE}).}

\item{dopb}{a Boolean value.
If \code{dopb = TRUE}, a progress bar is displayed.}
}
\value{
a list with the following elements: \itemize{
\item \code{g} a vector of size \code{n1 = length(grid)}.
Each component of this vector is an estimator of \eqn{g(x[i])}
where \code{x[i]} is the \eqn{i}-th element of the grid.

\item \code{best_a} a vector of the same size as \code{grid} indicating
for each value of the grid what is the optimal choice of \eqn{a} found by
our algorithm (which is used to estimate \eqn{g}).

\item \code{best_h} a vector of the same size as \code{grid} indicating
for each value of the grid what is the optimal choice of \eqn{h} found by
our algorithm (which is used to estimate \eqn{g}).

\item \code{first_step_g} first step estimator of \code{g}, computed using
the tuning parameters \code{best_a} and \code{h_firstStep[2]}.

\item \code{AMSE_estimated} an estimator of the part of the asymptotic MSE
that only depends on \eqn{a}.
}
}
\description{
A continuous elliptical distribution has a density of the form
\deqn{f_X(x) = {|\Sigma|}^{-1/2}
g\left( (x-\mu)^\top \, \Sigma^{-1} \, (x-\mu) \right),
}
where \eqn{x \in \mathbb{R}^d},
\eqn{\mu \in \mathbb{R}^d} is the mean,
\eqn{\Sigma} is a \eqn{d \times d} positive-definite matrix
and a function \eqn{g: \mathbb{R}_+ \rightarrow \mathbb{R}_+}, called the
density generator of \eqn{X}.
The goal is to estimate \eqn{g} at some point \eqn{\xi}, by
\deqn{
\widehat{g}_{n,h,a}(\xi)
:= \dfrac{\xi^{\frac{-d+2}{2}} \psi_a'(\xi)}{n h s_d}
\sum_{i=1}^n
  K\left( \dfrac{ \psi_a(\xi) - \psi_a(\xi_i) }{h} \right)
+ K\left( \dfrac{ \psi_a(\xi) + \psi_a(\xi_i) }{h} \right),
}
where
\eqn{s_d := \pi^{d/2} / \Gamma(d/2)},
\eqn{\Gamma} is the Gamma function,
\eqn{h} and \eqn{a} are tuning parameters (respectively the bandwidth and a
parameter controlling the bias at \eqn{\xi = 0}),
\eqn{\psi_a(\xi) := -a + (a^{d/2} + \xi^{d/2})^{2/d},}
\eqn{\xi \in \mathbb{R}}, \eqn{K} is a kernel function and
\eqn{\xi_i := (X_i - \mu)^\top \, \Sigma^{-1} \, (X_i - \mu),
}
for a sample \eqn{X_1, \dots, X_n}.
This function computes "optimal asymptotic" values for the bandwidth \eqn{h}
and the tuning parameter \eqn{a} from a first step bandwidth that the user
needs to provide.
}
\examples{
n = 500
d = 3
X = matrix(rnorm(n * d), ncol = d)
grid = seq(0, 5, by = 0.1)

result = EllDistrEst.adapt(X = X, grid = grid, h = 0.05)
plot(grid, result$g, type = "l")
lines(grid, result$first_step_g, col = "blue")

# Computation of true values
g = exp(-grid/2)/(2*pi)^{3/2}
lines(grid, g, type = "l", col = "red")

plot(grid, result$best_a, type = "l", col = "red")
plot(grid, result$best_h, type = "l", col = "red")

sum((g - result$g)^2, na.rm = TRUE) < sum((g - result$first_step_g)^2, na.rm = TRUE)

}
\references{
Ryan, V., & Derumigny, A. (2024).
On the choice of the two tuning parameters for nonparametric estimation of an
elliptical distribution generator
\href{https://arxiv.org/abs/2408.17087}{arxiv:2408.17087}.
}
\seealso{
\code{\link{EllDistrEst}} for the nonparametric estimation of the
elliptical distribution density generator,
\code{\link{EllDistrSim}} for the simulation of elliptical distribution samples.

\code{\link{estim_tilde_AMSE}} which is used in this function. It estimates
a component of the asymptotic mean-square error (AMSE) of the nonparametric
estimator of the elliptical density generator assuming \eqn{h} has been optimally
chosen.
}
\author{
Alexis Derumigny, Victor Ryan
}
