% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/KTMatrixEst.R
\name{KTMatrixEst}
\alias{KTMatrixEst}
\title{Fast estimation of Kendall's tau matrix}
\usage{
KTMatrixEst(dataMatrix, blockStructure = NULL, averaging = "no", N = NULL)
}
\arguments{
\item{dataMatrix}{matrix of size \code{(n,d)} containing \code{n} observations
of a \code{d}-dimensional random vector.}

\item{blockStructure}{list of vectors.
Each vector corresponds to one group of variables
and contains the indexes of the variables that belongs to this group.
\code{blockStructure} must be a partition of \code{1:d},
where \code{d} is the number of columns in \code{dataMatrix}.}

\item{averaging}{type of averaging used for fast estimation.
Possible choices are \itemize{
\item \code{no}: no averaging;
\item \code{all}: averaging all Kendall's taus in each block.
\code{N} is then the number of entries in the block, i.e. the
products of both dimensions.
\item \code{diag}: averaging along diagonal blocks elements.
\code{N} is then the minimum of the block's dimensions.
\item \code{row}: averaging Kendall's tau along the smallest block side.
\code{N} is then the minimum of the block's dimensions.
\item \code{random}: averaging Kendall's taus along a random sample of \code{N} entries
of the given block. These entries are chosen uniformly without replacement.
}}

\item{N}{number of entries to average (n the random case.
By default, \code{N} is then the minimum of the block's dimensions.}
}
\value{
matrix with dimensions depending on \code{averaging}.
\itemize{
\item If \code{averaging = no},
the function returns a matrix of dimension \code{(n,n)}
which estimates the Kendall's tau matrix.

\item Else, the function returns a matrix of dimension
\code{(length(blockStructure) , length(blockStructure))}
giving the estimates of the Kendall's tau for each block with ones on the diagonal.

}
}
\description{
Estimate Kendall's tau matrix using averaging estimators. Under
the structural assumption that Kendall's tau matrix is block-structured
with constant values in each off-diagonal block, this function estimates
Kendall's tau matrix ``fast'', in the sense that each interblock
coefficient is estimated in time \eqn{N \cdot n \cdot log(n)},
where \code{N} is the amount of pairs that are averaged.
}
\examples{
# Estimating off-diagonal block Kendall's taus
matrixCor = matrix(c(1  , 0.5, 0.3 ,0.3, 0.3,
                     0.5,   1, 0.3, 0.3, 0.3,
                     0.3, 0.3,   1, 0.5, 0.5,
                     0.3, 0.3, 0.5,   1, 0.5,
                     0.3, 0.3, 0.5, 0.5,   1), ncol = 5 , nrow = 5)
dataMatrix = mvtnorm::rmvnorm(n = 100, mean = rep(0, times = 5), sigma = matrixCor)
blockStructure = list(1:2, 3:5)
estKTMatrix = list()
estKTMatrix$all = KTMatrixEst(dataMatrix = dataMatrix,
                              blockStructure = blockStructure,
                              averaging = "all")
estKTMatrix$row = KTMatrixEst(dataMatrix = dataMatrix,
                              blockStructure = blockStructure,
                              averaging = "row")
estKTMatrix$diag = KTMatrixEst(dataMatrix = dataMatrix,
                               blockStructure = blockStructure,
                               averaging = "diag")
estKTMatrix$random = KTMatrixEst(dataMatrix = dataMatrix,
                                 blockStructure = blockStructure,
                                 averaging = "random", N = 2)
InterBlockCor = lapply(estKTMatrix, FUN = function(x) {sin(x[1,2] * pi / 2)})

# Estimation of the correlation between variables of the first group
# and of the second group
print(unlist(InterBlockCor))
# to be compared with the true value: 0.3.

}
\references{
van der Spek, R., & Derumigny, A. (2022).
Fast estimation of Kendall's Tau and conditional Kendall's Tau matrices under structural assumptions.
\href{https://arxiv.org/abs/2204.03285}{arxiv:2204.03285}.
}
\author{
Rutger van der Spek, Alexis Derumigny
}
\concept{Kendall correlation coefficient}
