\name{predDens}
\alias{predDens}
\title{Predictive posterior density of peak-over-threshold models}
\usage{
predDens(
  x,
  postsamp,
  threshold,
  type = c("continuous", "discrete"),
  extrapolation = FALSE,
  p,
  k,
  n
)
}
\arguments{
\item{x}{vector of length \code{r} containing the points at which to evaluate the density}

\item{postsamp}{a \code{m} by \code{2} matrix with columns containing the posterior samples of scale and shape parameters of the generalized Pareto distribution, respectively}

\item{threshold}{threshold for the generalized Pareto model, corresponding to the \eqn{n-k}th order statistic of the sample}

\item{type}{data type, either \code{"continuous"} or \code{"discrete"} for count data.}

\item{extrapolation}{logical; if \code{TRUE}, extrapolate using the threshold stability property}

\item{p}{scalar tail probability for the extrapolation. Must be smaller than \eqn{k/n} (only used if \code{extrapolation=TRUE})}

\item{k}{integer, number of exceedances for the generalized Pareto (only used if \code{extrapolation=TRUE})}

\item{n}{integer, number of observations in the full sample. Must be greater than \code{k}  (only used if \code{extrapolation=TRUE})}
}
\value{
a vector of length \code{r} of posterior predictive density values associated to \code{x}
}
\description{
Given posterior samples for the parameters of the continuous or discrete generalized Pareto distribution, return the predictive posterior density
of a peak above an intermediate or extreme threshold using the threshold stability property.
}
\examples{
\dontrun{
# generate data
set.seed(1234)
n <- 500
samp <- evd::rfrechet(n,0,3,4)
# set effective sample size and threshold
k <- 50
threshold <- sort(samp,decreasing = TRUE)[k+1]
# preliminary mle estimates of scale and shape parameters
mlest <- evd::fpot(samp, threshold)
# empirical bayes procedure
proc <- estPOT(
  samp,
  k = k,
  pn = c(0.01, 0.005),
  type = "continuous",
  method = "bayesian",
  prior = "empirical",
  start = as.list(mlest$estimate),
  sig0 = 0.1)
# predictive density estimation
yg <- seq(0, 50, by = 2)
nyg <- length(yg)
predDens_int <- predDens(
  yg,
  proc$post_sample,
  proc$t,
  "continuous",
  extrapolation = FALSE)
predDens_ext <- predDens(
  yg,
  proc$post_sample,
  proc$t,
  "continuous",
  extrapolation = TRUE,
  p = 0.001,
  k = k,
  n = n)
# plot
plot(
 x = yg,
 y = predDens_int,
 type = "l",
 lwd = 2,
 col = "dodgerblue",
 ylab = "",
 main = "Predictive posterior density")
lines(
 x = yg,
 y = predDens_ext,
 lwd = 2,
 col = "violet")
legend(
  "topright",
  legend = c("Intermediate threshold", "Extreme threshold"),
  lwd = 2,
  col = c("dodgerblue", "violet"))
  }
}
