% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/approx_kernel.R
\name{approx_kernel}
\alias{approx_kernel}
\title{Compute low-rank approximations(Nyström, Pivoted Cholesky, RFF)}
\usage{
approx_kernel(
  K = NULL,
  X = NULL,
  opt = c("nystrom", "pivoted", "rff"),
  kernel = c("gaussian", "laplace"),
  m = NULL,
  d,
  rho,
  eps = 1e-06,
  W = NULL,
  b = NULL,
  n_threads = 4
)
}
\arguments{
\item{K}{Exact Kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}. Used in \code{"nystrom"} and \code{"pivoted"}.}

\item{X}{Design matrix \eqn{X \in \mathbb{R}^{n \times d}}. Only required for \code{"rff"}.}

\item{opt}{Method for constructing or approximating :
\describe{
 \item{\code{"nystrom"}}{Construct a low-rank approximation of
     the kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
     using the Nyström approximation.}
 \item{\code{"pivoted"}}{Construct a low-rank approximation of
     the kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
     using Pivoted Cholesky decomposition.}
 \item{\code{"rff"}}{Construct a low-rank approximation of
     the kernel matrix \eqn{K \in \mathbb{R}^{n \times n}}
     using Random Fourier Features (RFF).}
}}

\item{kernel}{Kernel type either "gaussian"or "laplace".}

\item{m}{Approximation rank (number of random features) for the
low-rank kernel approximation. If not specified, the recommended
choice is
\deqn{\lceil n \cdot \log(d + 5) / 10 \rceil}
where \eqn{X} is design matrix, \eqn{n = nrow(X)} and \eqn{d = ncol(X)}.}

\item{d}{Design matrix's dimension (\eqn{d = ncol(X)}).}

\item{rho}{Scaling parameter of the kernel (\eqn{\rho}), specified by the user.}

\item{eps}{Tolerance parameter used only in \code{"pivoted"}
for stopping criterion of the Pivoted Cholesky decomposition.}

\item{W}{Random frequency matrix \eqn{\omega \in \mathbb{R}^{m \times d}}}

\item{b}{Random phase vector \eqn{b \in \mathbb{R}^m}, i.i.d. \eqn{\mathrm{Unif} [ 0,\,2\pi ]}.}

\item{n_threads}{Number of parallel threads.
The default is 4. If the system does not support 4 threads,
it automatically falls back to 1 thread. It is applied only for \code{opt = "nystrom"} or \code{opt = "rff"}
, and for the Laplace kernel (\code{kernel = "laplace"}).}
}
\value{
An S3 object of class \code{"approx_kernel"} containing the results of the
kernel approximation:

\itemize{
  \item \code{call}: The matched function call used to create the object.
  \item \code{opt}: The kernel approximation method actually used (\code{"nystrom", "pivoted", "rff"}).
  \item \code{K_approx}: \eqn{n \times n} approximated kernel matrix.
  \item \code{m}: Kernel approximation degree.
}

Additional components depend on the value of opt:


\strong{nystrom}

\itemize{
  \item \code{n_threads}: Number of threads used in the computation.
}

\strong{pivoted}

\itemize{
  \item \code{eps}: Numerical tolerance used for early stopping in the
                    pivoted Cholesky decomposition.
}

\strong{rff}

\itemize{
  \item \code{d}: Input design matrix's dimension.
  \item \code{rho}: Scaling parameter of the kernel.
  \item \code{W}: \eqn{m \times d} Random frequency matrix.
  \item \code{b}: Random phase \eqn{m}--vector.
  \item \code{used_supplied_Wb}: Logical; \code{TRUE} if user-supplied
        \code{W}, \code{b} were used, \code{FALSE} otherwise.
  \item \code{n_threads}: Number of threads used in the computation.
}
}
\description{
Computes low-rank kernel approximation \eqn{\tilde{K} \in \mathbb{R}^{n \times n}}using three methods:
Nyström approximation, Pivoted Cholesky decomposition, and
Random Fourier Features (RFF).
}
\details{
Requirements and what to supply:

\strong{Common}

\itemize{
  \item \code{d} and \code{rho} must be provided (non-\code{NULL}).
}

\strong{nystrom / pivoted}

\itemize{
  \item Require a precomputed kernel matrix \code{K}; error if \code{K} is \code{NULL}.
  \item If \code{m} is \code{NULL}, use \eqn{\lceil n \cdot \log(d + 5) / 10 \rceil}.
  \item For \code{"pivoted"}, a tolerance \code{eps} is used; the decomposition stops early
  when the next pivot (residual diagonal) drops below \code{eps}.
}

\strong{rff}

\itemize{
  \item \code{K} must be \code{NULL} (not used) and \code{X} must be provided with \code{d = ncol(X)}.
  \item The function automatically generates
        \code{W} (random frequency matrix \eqn{\omega \in \mathbb{R}^{m \times d}}) and
        \code{b} (random phase vector \eqn{b \in \mathbb{R}^{m}}).
  \item If the user provides them manually, both \code{W} and \code{b} must be specified and their dimensions must be compatible.
}
}
\examples{
# Data setting
set.seed(1)
d = 1
n = 1000
m = 50
X = matrix(runif(n*d, 0, 1), nrow = n, ncol = d)
y = as.vector(sin(2*pi*rowMeans(X)^3) + rnorm(n, 0, 0.1))
K = make_kernel(X, kernel = "gaussian", rho = 1)

# Example: RFF approximation
K_rff = approx_kernel(X = X, opt = "rff", kernel = "gaussian",
                      m = m, d = d, rho = 1,
                      n_threads = 1)

# Exapmle: Nystrom approximation
K_nystrom = approx_kernel(K = K, opt = "nystrom",
                          m = m, d = d, rho = 1,
                          n_threads = 1)

# Example: Pivoted Cholesky approximation
K_pivoted = approx_kernel(K = K, opt = "pivoted",
                          m = m, d = d, rho = 1)
}
