% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GaSPUtils.R
\name{gp.mcmc}
\alias{gp.mcmc}
\title{A wraper to fit a Gaussian stochastic process model with MCMC algorithms}
\usage{
gp.mcmc(
  obj,
  input.new = NULL,
  method = "Cauchy_prior",
  prior = list(),
  proposal = list(),
  nsample = 10000,
  verbose = TRUE
)
}
\arguments{
\item{obj}{an \code{S4} object \linkS4class{gp}}

\item{input.new}{a matrix of prediction locations. Default value is \code{NULL}, 
indicating that prediction is not carried out along with parameter estimation in the MCMC algorithm.}

\item{method}{a string indicating the Bayes estimation approaches with different choices of priors on correlation parameters:
\describe{
\item{Cauchy_prior}{This indicates that a fully Bayesian approach with objective priors is used
for parameter estimation, where location-scale parameters are assigned with constant priors and 
correlation parameters are assigned with half-Cauchy priors (default). 
If the smoothness parameter is estimated for \code{isotropic} covariance functions, the smoothness parameter is assigned with a uniform prior on (0, 4), 
indicating that the corresponding GP is at most four times mean-square differentiable. This is a 
reasonable prior belief for modeling spatial processes; If the smoothness parameter is estimated for \code{tensor} or \code{ARD} covariance functions,
the smoothness parameter is assigned with a uniform prior on (0, 6).}

\item{Ref_prior}{This indicates that a fully Bayesian approach with objective priors is used
for parameter estimation, where location-scale parameters are assigned with constant priors and 
correlation parameters are assigned with reference priors. 
If the smoothness parameter is estimated for \code{isotropic} covariance functions, the smoothness parameter is assigned with a uniform prior on (0, 4), 
indicating that the corresponding GP is at most four times mean-square differentiable. This is a 
reasonable prior belief for modeling spatial processes; If the smoothness parameter is estimated for \code{tensor} or \code{ARD} covariance functions,
the smoothness parameter is assigned with a uniform prior on (0, 6).}

\item{Beta_prior}{This indicates that a fully Bayesian approach with subjective priors is used
for parameter estimation, where location-scale parameters are assigned with constant priors and 
correlation parameters are assigned with \link{beta} priors parameterized as \eqn{Beta(a, b, lb, ub)}.
In the beta distribution, \strong{lb} and \strong{ub} are the support for correlation parameters, and
they should be determined based on domain knowledge. \strong{a} and \strong{b} are two shape parameters with default values at 1,
corresponding to the uniform prior over the support \eqn{(lb, ub)}. 
}
}}

\item{prior}{a list containing tuning parameters in prior distributions. This is used only if a Bayes estimation method with subjective priors is used.}

\item{proposal}{a list containing tuning parameters in proposal distributions. This is used only if a Bayes estimation method is used.}

\item{nsample}{an integer indicating the number of MCMC samples.}

\item{verbose}{a logical value. If it is \code{TRUE}, the MCMC progress bar is shown.}
}
\value{
a \code{\link{gp}} object with prior, proposal, MCMC samples included.
}
\description{
This function is a wraper to estimate parameters via MCMC algorithms in the GaSP model with different
choices of priors.
}
\examples{
 
 
code = function(x){
y = (sin(pi*x/5) + 0.2*cos(4*pi*x/5))*(x<=9.6) + (x/10-1)*(x>9.6) 
return(y)
}
n=100
input = seq(0, 20, length=n)
XX = seq(0, 20, length=99)
Ztrue = code(input)
set.seed(1234)
output = Ztrue + rnorm(length(Ztrue), sd=0.1)
obj = gp(formula=~1, output, input, 
        param=list(range=4, nugget=0.1,nu=2.5),
        smooth.est=FALSE,
        cov.model=list(family="matern", form="isotropic"))
        
fit.mcmc = gp.mcmc(obj, method="Cauchy_prior",
                   proposal=list(range=0.3, nugget=0.8),
                   nsample=100, verbose=TRUE)
                   
}
\seealso{
\link{GPBayes-package}, \code{\link{GaSP}}, \linkS4class{gp}, \code{\link{gp.optim}}
}
\author{
Pulong Ma \email{mpulong@gmail.com}
}
