% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SpatialScan.R
\name{SpatialScan}
\alias{SpatialScan}
\title{Spatial scan procedure}
\usage{
SpatialScan(
  method,
  data,
  sites_coord = NULL,
  system = NULL,
  mini = 1,
  maxi = nrow(sites_coord)/2,
  type_minimaxi = "sites/indiv",
  mini_post = NULL,
  maxi_post = NULL,
  type_minimaxi_post = "sites/indiv",
  sites_areas = NULL,
  MC = 999,
  typeI = 0.05,
  nbCPU = 1,
  variable_names = NULL,
  times = NULL
)
}
\arguments{
\item{method}{character vector. The scan procedures to apply on the data. Possible values are:
\itemize{
\item Univariate scan procedures: "UG" (univariate gaussian, see \code{\link{UG}}), "UNP" (univariate nonparametric, see \code{\link{UNP}})
\item Multivariate scan procedures: "MG" (multivariate gaussian, see \code{\link{MG}}), "MNP" (multivariate nonparametric, see \code{\link{MNP}})
\item Univariate functional scan procedures: "NPFSS" (nonparametric functional scan statistic, see \code{\link{NPFSS}}), "PFSS" (parametric functional scan statistic, see \code{\link{PFSS}}), "DFFSS" (distribution-free functional scan statistic, see \code{\link{DFFSS}}), "URBFSS" (univariate rank-based functional scan statistic, see \code{\link{URBFSS}})
\item Multivariate functional scan procedures: "NPFSS" (nonparametric functional scan statistic, see \code{\link{NPFSS}}), "MDFFSS" (multivariate distribution-free functional scan statistic, see \code{\link{MDFFSS}}), "MRBFSS" (multivariate rank-based functional scan statistic, see \code{\link{MRBFSS}}), "MPFSS", "MPFSS-LH", "MPFSS-W", "MPFSS-P" and "MPFSS-R" (parametric multivariate functional scan statistic ; "LH", "W", "P", "R" correspond respectively to the Lawley-Hotelling trace test statistic, The Wilks lambda test statistic, the Pillai trace test statistic and the Roy's maximum root test statistic, see \code{\link{MPFSS}}). Note that "MPFSS" computes "MPFSS-LH", "MPFSS-W", "MPFSS-P" and "MPFSS-R".
}}

\item{data}{list of numeric matrices or a matrix or a vector:
\itemize{
\item Univariate case: Vector of the data, each element corresponds to a site (or an individual if the observations are by individuals and not by sites).
\item Multivariate case: Matrix of the data, the rows correspond to the sites (or the individuals if the observations are by individuals and not by sites) and each column represents a variable.
\item Univariate functional case: Matrix of the data, the rows correspond to the sites (or to the individuals if the observations are by individuals and not by sites) and each column represents an observation time. The times must be the same for each site/individual. Depending on the scan procedure they also need to be equally-spaced.
\item Multivariate functional case: List of nb_sites (or nb_individuals if the observations are by individuals and not by sites) matrices of the data, the rows correspond to the variables and each column represents an observation time. The times must be the same for each site/individual. Depending on the scan procedure they also need to be equally-spaced.
}}

\item{sites_coord}{numeric matrix. Coordinates of the sites (or the individuals, in that case there can be many individuals with the same coordinates).}

\item{system}{character. System in which the coordinates are expressed: "Euclidean" or "WGS84".}

\item{mini}{numeric. A minimum for the clusters (see type_minimaxi). Changing the default value may bias the inference.}

\item{maxi}{numeric. A Maximum for the clusters (see type_minimaxi). Changing the default value may bias the inference.}

\item{type_minimaxi}{character. Type of minimum and maximum: by default "sites/indiv": the mini and maxi are on the number of sites or individuals in the potential clusters. Other possible values are "area": the minimum and maximum area of the clusters, or "radius": the minimum and maximum radius.}

\item{mini_post}{numeric. A minimum to filter the significant clusters a posteriori (see type_minimaxi_post). The default NULL is for no filtering with a a posteriori minimum.}

\item{maxi_post}{numeric. A maximum to filter the significant clusters a posteriori (see type_minimaxi_post). The default NULL is for no filtering with a a posteriori maximum.}

\item{type_minimaxi_post}{character. Type of minimum and maximum a posteriori: by default "sites/indiv": the mini_post and maxi_post are on the number of sites or individuals in the significant clusters. Other possible values are "area": the minimum and maximum area of the clusters, or "radius": the minimum and maximum radius.}

\item{sites_areas}{numeric vector. Areas of the sites. It must contain the same number of elements than the rows of sites_coord. If the data is on individuals and not on sites, there can be duplicated values. By default: NULL}

\item{MC}{numeric. Number of Monte-Carlo permutations to evaluate the statistical significance of the clusters. By default: 999.}

\item{typeI}{numeric. The desired type I error. A cluster will be evaluated as significant if its associated p-value is less than typeI. By default 0.05.}

\item{nbCPU}{numeric. Number of CPU. If nbCPU > 1 parallelization is done. By default: 1. Ignored for "UG" and "UNP"}

\item{variable_names}{character. Names of the variables. By default NULL. Ignored for the univariate and univariate functional scan procedures.}

\item{times}{numeric. Times of observation of the data. By default NULL. Ignored for the univariate and multivariate scan procedures.}
}
\value{
A list of objects of class ResScanOutput:
\itemize{
\item Univariate case (UG, UNP): A list of objects of class ResScanOutputUni
\item Multivariate case (MG, MNP): A list of objects of class ResScanOutputMulti
\item Univariate functional case (NPFSS, PFSS, DFFSS, URBFSS): A list of objects of class ResScanOutputUniFunct
\item Multivariate functional case (NPFSS, MPFSS, MDFFSS, MRBFSS): A list of objects of class ResScanOutputMultiFunct
}
}
\description{
This function computes the different scan procedures available in the package.
}
\examples{
# Univariate scan statistics
\donttest{
library(sp)
data("map_sites")
data("multi_data")
uni_data <- multi_data[,1]
coords <- coordinates(map_sites)
res <- SpatialScan(method = c("UG", "UNP"), data = uni_data, sites_coord = coords,
system = "WGS84", mini = 1, maxi = nrow(coords)/2)}
\dontshow{
library(sp)
data("map_sites")
data("multi_data")
uni_data <- multi_data[,1]
coords <- coordinates(map_sites)
res <- SpatialScan(method = c("UG", "UNP"), data = uni_data, sites_coord = coords,
system = "WGS84", mini = 1, maxi = nrow(coords)/2, MC = 9)}
# Multivariate scan statistics
\donttest{
library(sp)
data("map_sites")
data("multi_data")
coords <- coordinates(map_sites)
res <- SpatialScan(method = c("MG", "MNP"), data = multi_data, sites_coord = coords,
system = "WGS84", mini = 1, maxi = nrow(coords)/2)}
\dontshow{
library(sp)
data("map_sites")
data("multi_data")
indices <- c(51:75)
coords <- coordinates(map_sites[indices,])
res <- SpatialScan(method = c("MG", "MNP"), data = multi_data[indices,], sites_coord = coords,
system = "WGS84", mini = 1, maxi = nrow(coords)/2, MC = 9)
}
# Univariate functional scan statistics
\donttest{
library(sp)
data("map_sites")
data("funi_data")
coords <- coordinates(map_sites)
res <- SpatialScan(method = c("NPFSS", "PFSS", "DFFSS", "URBFSS"), data = funi_data,
sites_coord = coords, system = "WGS84", mini = 1, maxi = nrow(coords)/2)}
\dontshow{
library(sp)
data("map_sites")
data("funi_data")
indices <- c(51:75)
coords <- coordinates(map_sites[indices,])
res <- SpatialScan(method = c("NPFSS", "PFSS", "DFFSS", "URBFSS"), data = funi_data[indices,],
sites_coord = coords, system = "WGS84", mini = 1, maxi = 1, MC = 9)
}
# Multivariate functional
\donttest{
library(sp)
data("map_sites")
data("fmulti_data")
coords <- coordinates(map_sites)
res <- SpatialScan(method = c("NPFSS", "MPFSS", "MDFFSS", "MRBFSS"), data = fmulti_data,
sites_coord = coords, system = "WGS84", mini = 1, maxi = nrow(coords)/2)}
\dontshow{
library(sp)
data("map_sites")
data("fmulti_data")
indices <- c(51:75)
coords <- coordinates(map_sites[indices,])
res <- SpatialScan(method = c("NPFSS", "MPFSS", "MDFFSS", "MRBFSS"), data = fmulti_data[indices],
sites_coord = coords, system = "WGS84", mini = 1, maxi = 1, MC = 9)
}


}
\references{
For univariate scan statistics:
\itemize{
\item Inkyung Jung and Ho Jin Cho (2015). A Nonparametric Spatial Scan Statistic for Continuous Data. International Journal of Health Geographics, 14.
\item Martin Kulldorff and Lan Huang and Kevin Konty (2009). A Scan Statistic for Continuous Data Based on the Normal Probability Model. International Journal of Health Geographics, 8 (58).
}
For multivariate scan statistics:
\itemize{
\item Lionel Cucala and Michaël Genin and Florent Occelli and Julien Soula (2019). A Multivariate Nonparametric Scan Statistic for Spatial Data. Spatial statistics, 29, 1-14.
\item Lionel Cucala and Michaël Genin and Caroline Lanier and Florent Occelli (2017). A Multivariate Gaussian Scan Statistic for Spatial Data. Spatial Statistics, 21, 66-74.
}
For functional scan statistics:
\itemize{
\item Zaineb Smida and Lionel Cucala and Ali Gannoun. A Nonparametric Spatial Scan Statistic for Functional Data. Pre-print <https://hal.archives-ouvertes.fr/hal-02908496>.
\item Camille Frévent and Mohamed-Salem Ahmed and Matthieu Marbac and Michaël Genin. Detecting Spatial Clusters in Functional Data: New Scan Statistic Approaches. Pre-print <arXiv:2011.03482>.
\item Camille Frévent and Mohamed-Salem Ahmed and Sophie Dabo-Niang and Michaël Genin. Investigating Spatial Scan Statistics for Multivariate Functional Data. Pre-print <arXiv:2103.14401>.
}
}
\seealso{
\code{\link{ResScanOutput}}, \code{\link{ResScanOutputUni}}, \code{\link{ResScanOutputMulti}}, \code{\link{ResScanOutputUniFunct}} and \code{\link{ResScanOutputMultiFunct}}
}
