#' @title Diagnostic plots explaining K-means clustering and Fisher's exact test carried out by HDStIM
#'
#' @param mapped_data     Returned list from the \code{\link{HDStIM}} function.
#' @param path            Path to the folder to save figures generated by this function NULL by default.
#' @param verbose         Logical. To make function more verbose. Default is FALSE.
#' @import ggplot2
#' @return                A list of ggplot objects. If the path is not NULL, PNG files of the plots are saved in the specified folder.
#' @export
#' @examples
#' mapped_data <- HDStIM(chi11$expr_data, chi11$state_markers,
#'                   chi11$cluster_col, chi11$stim_label,
#'                   chi11$unstim_label, seed_val = 123, umap = FALSE, umap_cells = NULL,
#'                   verbose = FALSE)
#'
#' pk <- plot_K_Fisher(mapped_data, path = NULL, verbose = FALSE)
plot_K_Fisher <- function(mapped_data, path = NULL, verbose = FALSE){
  # Check if path exists; if not then create it.
  if(!is.null(path)){
    if(!dir.exists(path)){
      if(verbose){message(paste("Creating %s folder", path))}
      dir.create(path, recursive = TRUE)
    } else {
      if(verbose){message(paste(path, "folder already exists. Output will be over written."))}
    }
  }

  # Define colors.
  cbPalette <- c("#E69F00", "#56B4E9", "#009E73", "#F0E442", "#0072B2", "#D55E00", "#CC79A7")

  # Bind global variables.
  cell_population <- k_cluster <- stim_status <- stim_type <- cell_count_perc <- NULL

  # Group data according to clusters and stimulation type.
  group_data <- group_by(mapped_data$stacked_bar_plot_data, cell_population, stim_type)

  # Split groups into a list of individual tables.
  split_groups <- group_split(group_data)

  # Create ggplot for a single group.
  allplots <- list()
  for(i in 1:length(split_groups)){
    clust <- unique(split_groups[[i]]$cell_population)
    stim <- unique(split_groups[[i]]$stim_type)
    if(verbose){message(paste("Plotting for the cell population", clust, "and stimulation", stim))}
    stacked_plot <- ggplot(split_groups[[i]], aes(fill = k_cluster, y=cell_count_perc, x=stim_status)) +
      geom_bar(position="fill", stat="identity") +
      scale_y_continuous(labels=scales::percent) +
      scale_fill_manual(values=cbPalette) +
      labs(title=paste0("Cell Pop.: ", clust,
                        "; Stim.: ", stim,
                        "\nFisher's P-value: ", round(unique(split_groups[[i]]$f_p_val),3),
                        "; Stim. Cluster: ",unique(split_groups[[i]]$stim_clust)),
           x ="Stimulation Status", y = "Cell Count %", fill = "K-means Cluster")

    allplots[[i]] <- stacked_plot

    if (!is.null(path)) {
      ggsave(
        paste0("plot_KF_", clust, "_", stim, ".png"),
        plot = stacked_plot,
        device = "png",
        path = file.path(path),
        width = 6,
        height = 4,
        dpi = 300
      )
    }
  }
  return(allplots)
}




