% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ICS_S3.R
\name{ICS-S3}
\alias{ICS-S3}
\alias{ICS}
\title{Two Scatter Matrices ICS Transformation}
\usage{
ICS(
  X,
  S1 = ICS_cov,
  S2 = ICS_cov4,
  S1_args = list(),
  S2_args = list(),
  algorithm = c("whiten", "standard", "QR"),
  center = FALSE,
  fix_signs = c("scores", "W"),
  na.action = na.fail
)
}
\arguments{
\item{X}{a numeric matrix or data frame containing the data to be
transformed.}

\item{S1}{a numeric matrix containing the first scatter matrix, an object
of class \code{"ICS_scatter"} (that typically contains the location vector
and scatter matrix as \code{location} and \code{scatter} components), or a
function that returns either of those options. The default is function
\code{\link{ICS_cov}()} for the sample covariance matrix.}

\item{S2}{a numeric matrix containing the second scatter matrix, an object
of class \code{"ICS_scatter"} (that typically contains the location vector
and scatter matrix as \code{location} and \code{scatter} components), or a
function that returns either of those options. The default is function
\code{\link{ICS_cov4}()} for the covariance matrix based on fourth order
moments.}

\item{S1_args}{a list containing additional arguments for \code{S1} (only
relevant if \code{S1} is a function).}

\item{S2_args}{a list containing additional arguments for \code{S2} (only
relevant if \code{S2} is a function).}

\item{algorithm}{a character string specifying with which algorithm
the invariant coordinate system is computed. Possible values are
\code{"whiten"}, \code{"standard"} or \code{"QR"}.
See \sQuote{Details} for more information.}

\item{center}{a logical indicating whether the invariant coordinates should
be centered with respect to first locattion or not (default to \code{FALSE}).
Centering is only applicable if the
first scatter object contains a location component, otherwise this is set to
\code{FALSE}. Note that this only affects the scores of the invariant
components (output component \code{scores}), but not the generalized
kurtosis values (output component \code{gen_kurtosis}).}

\item{fix_signs}{a character string specifying how to fix the signs of the
invariant coordinates. Possible values are \code{"scores"} to fix the signs
based on (generalized) skewness values of the coordinates, or \code{"W"} to
fix the signs based on the coefficient matrix of the linear transformation.
See \sQuote{Details} for more information.}

\item{na.action}{a function to handle missing values in the data (default
to \code{\link[stats]{na.fail}}, see its help file for alternatives).}
}
\value{
An object of class \code{"ICS"} with the following components:
\item{gen_kurtosis}{a numeric vector containing the generalized kurtosis
values of the invariant coordinates.}
\item{W}{a numeric matrix in which each row contains the coefficients of the
linear transformation to the corresponding invariant coordinate.}
\item{scores}{a numeric matrix in which each column contains the scores of
the corresponding invariant coordinate.}
\item{gen_skewness}{a numeric vector containing the (generalized) skewness
values of the invariant coordinates (only returned if
\code{fix_signs = "scores"}).}
\item{S1_label}{a character string providing a label for the first scatter
matrix to be used by various methods.}
\item{S2_label}{a character string providing a label for the second scatter
matrix to be used by various methods.}
\item{S1_args}{a list containing additional arguments used to compute
\code{S1} (if a function was supplied).}
\item{S2_args}{a list containing additional arguments used to compute
\code{S2} (if a function was supplied).}
\item{algorithm}{a character string specifying how the invariant
coordinate is computed.}
\item{center}{a logical indicating whether or not the data were centered
with respect to the first location vector before computing the invariant
coordinates.}
\item{fix_signs}{a character string specifying how the signs of the
invariant coordinates were fixed.}
}
\description{
Transforms the data via two scatter matrices to an invariant coordinate
system or independent components, depending on the underlying assumptions.
Function \code{ICS()} is intended as a replacement for \code{\link{ics}()}
and \code{\link{ics2}()}, and it combines their functionality into a single
function. Importantly, the results are returned as an
\code{\link[base:class]{S3}} object rather than an
\code{\link[methods:Classes_Details]{S4}} object. Furthermore, \code{ICS()}
implements recent improvements, such as a numerically stable algorithm based
on the QR algorithm for a common family of scatter pairs.
}
\details{
For a given scatter pair \eqn{S_{1}}{S1} and \eqn{S_{2}}{S2}, a matrix
\eqn{Z} (in which the columns contain the scores of the respective invariant
coordinates) and a matrix \eqn{W} (in which the rows contain the
coefficients of the linear transformation to the respective invariant
coordinates) are found such that:
\itemize{
\item The columns of \eqn{Z} are whitened with respect to
\eqn{S_{1}}{S1}. That is, \eqn{S_{1}(Z) = I}{S1(Z) = I}, where \eqn{I}
denotes the identity matrix.
\item The columns of \eqn{Z} are uncorrelated with respect to
\eqn{S_{2}}{S2}. That is, \eqn{S_{2}(Z) = D}{S2(Z) = D}, where \eqn{D}
is a diagonal matrix.
\item The columns of \eqn{Z} are ordered according to their generalized
kurtosis.
}

Given those criteria, \eqn{W} is unique up to sign changes in its rows. The
argument \code{fix_signs} provides two ways to ensure uniqueness of \eqn{W}:
\itemize{
\item If argument \code{fix_signs} is set to \code{"scores"}, the signs
in \eqn{W} are fixed such that the generalized skewness values of all
components are positive. If \code{S1} and \code{S2} provide location
components, which are denoted by \eqn{T_{1}}{T1} and \eqn{T_{2}}{T2},
the generalized skewness values are computed as
\eqn{T_{1}(Z) - T_{2}(Z)}{T1(Z) - T2(Z)}.
Otherwise, the skewness is computed by subtracting the column medians of
\eqn{Z} from the corresponding column means so that all components are
right-skewed. This way of fixing the signs is preferred in an invariant
coordinate selection framework.
\item If argument \code{fix_signs} is set to \code{"W"}, the signs in
\eqn{W} are fixed independently of \eqn{Z} such that the maximum element
in each row of \eqn{W} is positive and that each row has norm 1. This is
the usual way of fixing the signs in an independent component analysis
framework.
}

In principal, the order of \eqn{S_{1}}{S1} and \eqn{S_{2}}{S2} does not
matter if both are true scatter matrices. Changing their order will just
reverse the order of the components and invert the corresponding
generalized kurtosis values.

The same does not hold when at least one of them is a shape matrix rather
than a true scatter matrix. In that case, changing their order will also
reverse the order of the components, but the ratio of the generalized
kurtosis values is no longer 1 but only a constant. This is due to the fact
that when shape matrices are used, the generalized kurtosis values are only
relative ones.

Different algorithms are available to compute the invariant coordinate
system of a data frame \eqn{X_n} with \eqn{n} observations:
\itemize{
\item \strong{"whiten"}: whitens the data \eqn{X_n} with respect to the first
scatter matrix before computing the second scatter matrix. If \code{S2} is not a function, whitening is not applicable.
\itemize{
\item whiten the data \eqn{X_n} with respect to the first
scatter matrix: \eqn{Y_n =  X_n S_1(X_n)^{-1/2}}
\item compute \eqn{S_2} for the uncorrelated data: \eqn{S_2(Y_n)}
\item perform the eigendecomposition of \eqn{S_2(Y_n)}: \eqn{S_2(Y_n) = UDU'}
\item compute \eqn{W}: \eqn{W = U' S_1(X_n)^{-1/2}}
}
\item \strong{"standard"}: performs the spectral decomposition of the
symmetric matrix  \eqn{M(X_n)}
\itemize{
\item compute \eqn{M(X_n) = S_1(X_n)^{-1/2} S_2(X_n) S_1(X_n)^{-1/2}}
\item perform the eigendecomposition of \eqn{M(X_n)}: \eqn{M(X_n) = UDU'}
\item compute \eqn{W}: \eqn{W = U' S_1(X_n)^{-1/2}}
}
\item \strong{"QR"}: numerically stable algorithm based on the QR algorithm for a
common family of scatter pairs: if \code{S1} is \code{\link{ICS_cov}()}
or \code{\link[stats]{cov}()}, and if \code{S2} is one of
\code{\link{ICS_cov4}()}, \code{\link{ICS_covW}()}
, \code{\link{ICS_covAxis}()}, \code{\link{cov4}()},
\code{\link{covW}()}, or \code{\link{covAxis}()}.
For other scatter pairs, the QR algorithm is not
applicable.   See Archimbaud et al. (2023)
for details. Note that the QR algorithm requires \proglang{LAPACK} version
\eqn{\geq}{>=} 3.11.0 to attain better numerical stability.
}

The "whiten" algorithm is the most natural version and therefore the default. The option "standard"
should be only used if the scatters provided are not functions but precomputed matrices.
The option "QR" is mainly of interest when there are numerical issues when "whiten" is used and the
scatter combination allows its usage.

Note that when the purpose of ICS is outlier detection the package \link[ICSOutlier]{ICSOutlier}
provides additional functionalities as does the package \code{ICSClust} in case the
goal of ICS is dimension reduction prior clustering.
}
\examples{
# import data
data("iris")
X <- iris[,-5]

# run ICS
out_ICS <- ICS(X)
out_ICS
summary(out_ICS)

# extract generalized eigenvalues
gen_kurtosis(out_ICS)
# Plot
screeplot(out_ICS)

# extract the components
components(out_ICS)
components(out_ICS, select = 1:2)

# Plot
plot(out_ICS)

# equivalence with previous functions
out_ics <- ics(X, S1 = cov, S2 = cov4, stdKurt = FALSE)
out_ics
out_ics2 <- ics2(X, S1 = MeanCov, S2 = Mean3Cov4)
out_ics2
out_ICS


# example using two functions
X1 <- rmvnorm(250, rep(0,8), diag(c(rep(1,6),0.04,0.04)))
X2 <- rmvnorm(50, c(rep(0,6),2,0), diag(c(rep(1,6),0.04,0.04)))
X3 <- rmvnorm(200, c(rep(0,7),2), diag(c(rep(1,6),0.04,0.04)))
X.comps <- rbind(X1,X2,X3)
A <- matrix(rnorm(64),nrow=8)
X <- X.comps \%*\% t(A)
ics.X.1 <- ICS(X)
summary(ics.X.1)
plot(ics.X.1)
# compare to
pairs(X)
pairs(princomp(X,cor=TRUE)$scores)


# slow:
if (require("ICSNP")) {
  ics.X.2 <- ICS(X, S1 = tyler.shape, S2 = duembgen.shape,
  S1_args = list(location=0))
  summary(ics.X.2)
  plot(ics.X.2)
  # example using three pictures
  library(pixmap)
  fig1 <- read.pnm(system.file("pictures/cat.pgm", package = "ICS")[1],
                   cellres = 1)
  fig2 <- read.pnm(system.file("pictures/road.pgm", package = "ICS")[1],
                   cellres = 1)
  fig3 <- read.pnm(system.file("pictures/sheep.pgm", package = "ICS")[1],
                   cellres = 1)
  p <- dim(fig1@grey)[2]
  fig1.v <- as.vector(fig1@grey)
  fig2.v <- as.vector(fig2@grey)
  fig3.v <- as.vector(fig3@grey)
  X <- cbind(fig1.v, fig2.v, fig3.v)
  A <- matrix(rnorm(9), ncol = 3)
  X.mixed <- X \%*\% t(A)
  ICA.fig <- ICS(X.mixed)
  par.old <- par()
  par(mfrow=c(3,3), omi = c(0.1,0.1,0.1,0.1), mai = c(0.1,0.1,0.1,0.1))
  plot(fig1)
  plot(fig2)
  plot(fig3)
  plot(pixmapGrey(X.mixed[,1], ncol = p, cellres = 1))
  plot(pixmapGrey(X.mixed[,2], ncol = p, cellres = 1))
  plot(pixmapGrey(X.mixed[,3], ncol = p, cellres = 1))
  plot(pixmapGrey(components(ICA.fig)[,1], ncol = p, cellres = 1))
  plot(pixmapGrey(components(ICA.fig)[,2], ncol = p, cellres = 1))
  plot(pixmapGrey(components(ICA.fig)[,3], ncol = p, cellres = 1))
}
}
\references{
Tyler, D.E., Critchley, F., Duembgen, L. and Oja, H. (2009) Invariant
Co-ordinate Selection. \emph{Journal of the Royal Statistical Society,
Series B}, \bold{71}(3), 549--592. \doi{10.1111/j.1467-9868.2009.00706.x}.

Archimbaud, A., Drmac, Z., Nordhausen, K., Radojcic, U. and Ruiz-Gazen, A.
(2023) Numerical Considerations and a New Implementation for Invariant
Coordinate Selection. \emph{SIAM Journal on Mathematics of Data Science},
\bold{5}(1), 97--121. \doi{10.1137/22M1498759}.
}
\seealso{
\code{\link{gen_kurtosis}()}, \code{\link[=coef.ICS]{coef}()},
\code{\link{components}()}, \code{\link[=fitted.ICS]{fitted}()}, and
\code{\link[=plot.ICS]{plot}()} methods
}
\author{
Andreas Alfons and Aurore Archimbaud, based on code for
\code{\link{ics}()} and \code{\link{ics2}()} by Klaus Nordhausen
}
