% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/residuals.R
\name{residuals}
\alias{residuals}
\alias{residuals.lmm}
\alias{residuals.clmm}
\alias{residuals.mlmm}
\title{Extract The Residuals From a Linear Mixed Model}
\usage{
\method{residuals}{lmm}(
  object,
  type = "response",
  variable = NULL,
  at = NULL,
  newdata = NULL,
  p = NULL,
  format = "long",
  keep.data = FALSE,
  fitted.ci = FALSE,
  simplify = TRUE,
  var,
  ...
)

\method{residuals}{clmm}(object, ...)

\method{residuals}{mlmm}(object, simplify = TRUE, ...)
}
\arguments{
\item{object}{a \code{lmm} object.}

\item{type}{[character] type of residual to output such as raw residuals (\code{"response"}), normalized residuals (\code{"normalized"}, partial residuals (\code{"partial"}).
Can also be \code{"all"} to output all except partial residuals. See detail section.}

\item{variable}{[character vector] name of the variable relative to which the partial residuals should be computed.}

\item{at}{[data.frame] values for the covariates at which to evaluate the partial residuals.}

\item{newdata}{[data.frame] dataset relative to which the residuals should be computed. Only relevant if differs from the dataset used to fit the model.}

\item{p}{[numeric vector] value of the model coefficients at which to evaluate the residuals. Only relevant if differs from the fitted values.}

\item{format}{[character] Should the residuals be output
in a matrix format with clusters in row and timepoints in columns (\code{"wide"}),
or in a data.frame/vector with as many rows as observations (\code{"long"})}

\item{keep.data}{[logical] Should the dataset relative to which the residuals are evaluated be output along side the residual values?
Only possible in the long format.}

\item{fitted.ci}{[logical] Should the confidence intervals relative to the fitted values be added to the output. Only relevant when argument \code{keep.data=TRUE}.}

\item{simplify}{[logical] Simplify the data format (vector instead of data.frame) and column names (no mention of the time variable) when possible.
Otherwise, information about the call and reference values used for partial residuals be added as an attribute.}

\item{var}{[Deprecated] Not used anymore, replaced by argument variable.}

\item{...}{Not used. For compatibility with the generic method.}
}
\value{
\bold{lmm}: a vector or a data.frame when \code{format="long"} (one line per observation, one column per type of residual),
a matrix when \code{format="wide"}  (one line per cluster, one column per timepoint).
}
\description{
Extract or compute the residuals of a linear mixed model.
}
\details{
The argument \code{type} defines how the residuals are computed:
\itemize{
\item \code{"response"}: raw residual, i.e. observed outcome minus fitted value \eqn{\varepsilon_{ij} = Y_{ij} - X_{ij} \hat{\beta}}.
\item \code{"pearson"}: each raw residual is divided by its modeled standard deviation \eqn{\varepsilon_{ij} = \frac{Y_{ij} - X_{ij} \hat{\beta}}{\sqrt{\hat{\omega}_{ij}}}}.
\item \code{"studentized"}: same as \code{"pearson"} but excluding the contribution of the cluster in the modeled standard deviation  \eqn{\varepsilon_{ij} = \frac{Y_{ij} - X_{ij} \hat{\beta}}{\sqrt{\hat{\omega}_{ij}-\hat{q}_{ij}}}}.
\item \code{"normalized"}: raw residuals are multiplied, within clusters, by the inverse of the (upper) Cholesky factor of the modeled residual variance covariance matrix \eqn{\varepsilon_{ij} = ( Y_{i} - X_{i} \hat{\beta} )\hat{C}^{-1}}.
\item \code{"normalized2"}: raw residuals are multiplied, within clusters, by the inverse of the modeled residual variance covariance matrix \eqn{\varepsilon_{ij} = ( Y_{i} - X_{i} \hat{\beta} )\hat{Omega}^{-1}}.
\item \code{"scaled"}: scaled residuals (see PROC MIXED in SAS). Numerically identical to \code{"normalized"} but computed by sequentially scaling and centering the residuals, to make them conditionally independent of previous residuals from the same cluster at previous repetitions.
\item \code{"partial"}: partial residuals (\eqn{\gamma E + \hat{\varepsilon}}). A reference level can be also be specified via the attribute \code{"reference"} to change the absolute level of the partial residuals.
\code{"partial-center"}: partial residuals with centered continuous covariates (\eqn{\gamma E + \hat{\varepsilon}} where \eqn{E} has been centered, i.e., has 0-mean)
}
where
\itemize{
\item \eqn{X=(E,W)} the design matrix. For partial residuals, it is split according to the variable(s) in argument \code{variable} (\eqn{E}) and the rest (\eqn{W}).
\item \eqn{Y} the outcome
\item \eqn{\hat{\beta}=(\hat{\gamma},\hat{\delta})} the estimated mean coefficients relative to \eqn{X=(E,W)}
\item \eqn{\hat{\Omega}} the modeled variance-covariance of the residuals and \eqn{\hat{\omega}} its diagonal elements
\item \eqn{\hat{C}} the upper Cholesky factor of \eqn{\hat{\Omega}}, i.e. upper triangular matrix satisfying \eqn{\hat{C}^{t} \hat{C} = \hat{\Omega}}
\item \eqn{\hat{Q}_i= X_i (X^{t}\hat{\Omega}X)^{-1}X_i^{t}} a cluster specific correction factor, approximating the contribution of cluster i to \eqn{\hat{\Omega}}. Its diagonal elements are denoted \eqn{\hat{q}_i}.
\item \eqn{\hat{D}_i} the upper Cholesky factor of \eqn{\hat{\Omega}-\hat{Q}_i}
}

Setting argument \code{fitted.ci} to \code{TRUE}, \code{simplify} to \code{FALSE}, \code{format} to \code{"long"} returns an attribute \code{"grad"} containing the first order partial derivatives of the residuals with respect to the model parameters.
}
\examples{

#### simulate data in the long format ####
set.seed(10)
dL <- sampleRem(100, n.times = 3, format = "long")

#### Linear Model ####
e.lm <- lmm(Y ~ visit + X1 + X2 + X6, data = dL)

## partial residuals
pRes <- residuals(e.lm, type = "partial", variable = "X6")
range(residuals(e.lm) + dL$X6 * coef(e.lm)["X6"] - pRes)
e.reslm <- residuals(e.lm, type = "partial", variable = "X6", keep.data = TRUE, simplify = FALSE)
plot(e.reslm)

## partial residuals with specific reference
residuals(e.lm, type = "partial", variable = "X1",
          at = data.frame(visit=factor(2,1:3),X2=0,X6=3))

## partial residuals with centered covariates
residuals(e.lm, type = "partial-center", variable = "X1")

#### Linear Mixed Model ####
eUN.lmm <- lmm(Y ~ visit + X1 + X2 + X5 + X6,
               repetition = ~visit|id, structure = "UN", data = dL)

## residuals
e.resL <- residuals(eUN.lmm, type = "normalized",
                    keep.data = TRUE, simplify = FALSE)
plot(e.resL, type = "qqplot")
plot(e.resL, type = "scatterplot", labeller = ggplot2::label_both)
e.resW <- residuals(eUN.lmm, format = "wide", type = "normalized",
                    simplify = FALSE)
plot(e.resW, type = "correlation")

## residuals and predicted values
residuals(eUN.lmm, type = "all")
residuals(eUN.lmm, type = "all", keep.data = TRUE)

## partial residuals
residuals(eUN.lmm, type = "partial", variable = c("(Intercept)","visit","X6"))
}
\keyword{methods}
