#' Longitudinal Attribution of Disability and Death Based on Sullivan Method
#'
#' @description
#' Computes the relative and/or absolute contributions of causes (i.e., covariates) to disability and death across a specified age range,
#' based on a copula regression models with semiparametric additive hazards margins object. This attribution approach adopts a Sullivan method
#' for cohort health expectancy.
#'
#' @name Attribution_sullivan
#' @aliases Attribution_sullivan
#'
#' @param object A fitted \code{LongDecompHE} object returned by \code{copula_additive}.
#' @param type.attrib Type of attribution output to return; can be one of:
#' \itemize{
#'   \item \code{"rel"}: Relative contributions of causes over age (i.e., time);
#'   \item \code{"abs"}: Absolute contributions of causes over age (i.e., time);
#'   \item \code{"both"}: Both relative and absolute contributions (default).
#' }
#'
#' @return A list containing the following components:
#' \describe{
#'   \item{Relative_Contributions_1}{Matrix of relative contributions to disability prevalence by cause and age}
#'   \item{Relative_Contributions_2}{Matrix of relative contributions to death probability by cause and time}
#'   \item{Absolute_Contributions_1}{Matrix of absolute contributions to disability prevalence by cause and age}
#'   \item{Absolute_Contributions_2}{Matrix of absolute contributions to death probability by cause and time}
#'   \item{var_list}{Vector of covariate names used in the model}
#'   \item{time_list}{Vector of attribution ages}
#'   \item{copula}{Character indicating attribution type ("Attribution_sullivan")}
#' }
#'#' Additional components \code{copula} and \code{summary} are included for compatibility with generic
#' @details
#' This function loops over all attribution-eligible ages in the fitted model (from 0 to \code{floor(u-1)}),
#' Use \code{summary()} on the returned object to print the attribution tables.
#'
#' @seealso \code{\link{copula_additive}} for model fitting;
#' \code{\link{summary.LongDecompHE}} for output methods.
#'
#' @importFrom stats printCoefmat
#' @importFrom stats setNames
#' @export
#'
#' @examples
#' \donttest{
#' # Fit a model (see copula_additive)
#' data(simulated_dataA)
#' u1 = u2 = max(simulated_dataA$visit_time)
#' var_list = c("Z1", "Z2", "Z3")
#' copula_additive_model <-  copula_additive(data = simulated_dataA,
#'                                         var_list = var_list,
#'                                         l1=0, u1 = u1, m1 = 3,
#'                                         l2=0, u2 = u2, m2 = 3,
#'                                         method = "combined", iter=1000,
#'                                         stepsize=1e-6,
#'                                         hes = TRUE,
#'                                         control = list(maxit = 10000))
#' summary(copula_additive_model)
#' # Attribution analysis (both relative and absolute)
#' attributionA <- Attribution_sullivan(object = copula_additive_model, type.attrib = "both")
#' summary(attributionA)
#'
#' # Relative only
#' attributionA_rel <- Attribution_sullivan(object = copula_additive_model, type.attrib = "rel")
#'
#' # Absolute only
#' attributionA_abs <- Attribution_sullivan(object = copula_additive_model, type.attrib = "abs")
#' }


Attribution_sullivan <- function(object, type.attrib="both"){
  max_time <- floor(object$u)-1
  var_list <- object$var_list
  time_list <- grep("^time_", names(object$data), value = TRUE)

  attributionResults_disability <- Longitudinal_disability_attribution_pai(object = object, "both", 0)
  attributionResults_death <- Longitudinal_death_attribution(object = object, "both", 0)

  Relative_Contributions_1 <- attributionResults_disability$att_final$att.rel
  Absolute_Contributions_1 <- attributionResults_disability$att_final$att.abs

  Relative_Contributions_2 <- attributionResults_death$att_final$att.rel
  Absolute_Contributions_2 <- attributionResults_death$att_final$att.abs

  for (i in (1:max_time)){
    attributionResults_disability <- Longitudinal_disability_attribution_pai(object = object, "both", i)
    attributionResults_death <- Longitudinal_death_attribution(object = object, "both", i)
    Relative_Contributions_1 <- cbind(Relative_Contributions_1,attributionResults_disability$att_final$att.rel)
    Absolute_Contributions_1 <- cbind(Absolute_Contributions_1,attributionResults_disability$att_final$att.abs)

    Relative_Contributions_2 <- cbind(Relative_Contributions_2,attributionResults_death$att_final$att.rel)
    Absolute_Contributions_2 <- cbind(Absolute_Contributions_2,attributionResults_death$att_final$att.abs)
  }
  colnames(Relative_Contributions_1) <- c(1:(max_time+1))
  colnames(Absolute_Contributions_1) <- c(1:(max_time+1))

  colnames(Relative_Contributions_2) <- c(1:(max_time+1))
  colnames(Absolute_Contributions_2) <- c(1:(max_time+1))

  Attribution <- switch(type.attrib,
                        "rel" = list(Relative_Contributions_1 = Relative_Contributions_1, Relative_Contributions_2 = Relative_Contributions_2),
                        "abs" = list(Absolute_Contributions_1 = Absolute_Contributions_1, Absolute_Contributions_2 = Absolute_Contributions_2),
                        "both" =  list(Relative_Contributions_1 = Relative_Contributions_1,
                                       Relative_Contributions_2 = Relative_Contributions_2,
                                       Absolute_Contributions_1 = Absolute_Contributions_1,
                                       Absolute_Contributions_2 = Absolute_Contributions_2))

  Attribution$var_list <- var_list
  Attribution$time_list <- 0:max_time
  Attribution$copula <- "Attribution_sullivan"

  class(Attribution) <- "LongDecompHE"
  return(Attribution)

}

