#' Two-dimensional decomposition of the difference between the two cohort HE by age and cause
#' @description
#' This function implements the two-dimensional stepwise replacement decomposition algorithm (Andreev et al., 2002)
#' to quantify age-cause-specific effect of mortality and disability to the difference in cohort
#' health expectancy between two populations, based on longitudinal attribution results.
#'
#'
#' @name Decomp_sullivan
#' @aliases Decomp_sullivan
#' @param attribution1 A list returned by \code{Attribution_sullivan()}, representing Population 1.
#' @param attribution2 A list returned by \code{Attribution_sullivan()}, representing Population 2.
#'
#' @return A list including matrices of mortality effects, disability effects, and total effects,
#'         as well as cohort HE and LE for each population.
#'
#'
#' @examples
#' data(attributionA)
#' data(attributionB)
#' summary(attributionA)
#' summary(attributionB)
#' decom_results <- Decomp_sullivan(attribution1=attributionA,attribution2=attributionA)
#' summary(decom_results)
#'
#'
#' @export
#'
#'

Decomp_sullivan <- function(attribution1,attribution2){
  if (is.null(attribution1$Absolute_Contributions_1)) {
    stop("Input data missing Absolute_Contributions_1: please ensure type.attrib is set to 'both' or 'abs'")
  }

  t <- t1 <- ncol(attribution1$Absolute_Contributions_1)
  t2 <- ncol(attribution2$Absolute_Contributions_1)
  if (t1 != t2) {
    stop("Please input attribution results with the same number of age groups")
  }

  # Life table for Population 1
  lx_1 <- matrix(1,nrow=1,ncol=(t+1))
  Lx_1 <- matrix(0,nrow=1,ncol=t)
  Lxh_1 <- matrix(0,nrow=1,ncol=t)
  Tx_1 <- matrix(0,nrow=1,ncol=t)
  Txh_1 <- matrix(0,nrow=1,ncol=t)
  ex_1 <- matrix(0,nrow=1,ncol=t)
  Hx_1 <- matrix(0,nrow=1,ncol=(t+1))

  # Life table for Population 2
  lx_2 <- matrix(1,nrow=1,ncol=(t+1))
  Lx_2 <- matrix(0,nrow=1,ncol=t)
  Lxh_2 <- matrix(0,nrow=1,ncol=t)
  Tx_2 <- matrix(0,nrow=1,ncol=t)
  Txh_2 <- matrix(0,nrow=1,ncol=t)
  ex_2 <- matrix(0,nrow=1,ncol=t)
  Hx_2 <- matrix(0,nrow=1,ncol=(t+1))
  mortality_Andreev <- matrix(0,nrow=nrow(attribution1$Absolute_Contributions_1),ncol=t)
  disability_Andreev <- matrix(0,nrow=nrow(attribution1$Absolute_Contributions_1),ncol=t)
  for (i in c(1:t)){
    lx_1[i+1]=lx_1[i]*(1-attribution1$Absolute_Contributions_2[1,i])
    Lx_1[i] = (lx_1[i]+lx_1[i+1])/2
    Lxh_1[i]=(lx_1[i]+lx_1[i+1])/2*(1-attribution1$Absolute_Contributions_1[1,i])

    lx_2[i+1]=lx_2[i]*(1-attribution2$Absolute_Contributions_2[1,i])
    Lx_2[i] = (lx_2[i]+lx_2[i+1])/2
    Lxh_2[i]=(lx_2[i]+lx_2[i+1])/2*(1-attribution2$Absolute_Contributions_1[1,i])
  }
  for (i in c(1:t)){
    Tx_1[i] <- sum(Lx_1[i:t])
    Txh_1[i] <- sum(Lxh_1[i:t])
    ex_1[i] <- Tx_1[i]/lx_1[i]
    Hx_1[i] <- Txh_1[i]/lx_1[i]

    Tx_2[i] <- sum(Lx_2[i:t])
    Txh_2[i] <- sum(Lxh_2[i:t])
    ex_2[i] <- Tx_2[i]/lx_2[i]
    Hx_2[i] <- Txh_2[i]/lx_2[i]
  }
  Hx_1[t+1] = 0
  Hx_2[t+1] = 0

  for (i in c(1:t)){
    for(k in c(2:nrow(attribution1$Absolute_Contributions_1))){

      if(k==2){
        mortality_Andreev[k,i] = 0.5 * (attribution1$Absolute_Contributions_2[k,i]-attribution2$Absolute_Contributions_2[k,i])*
          (
            lx_2[i] *( (1-sum(attribution1$Absolute_Contributions_1[(k+1):nrow(attribution1$Absolute_Contributions_1),i]))/2
                       - (attribution1$Absolute_Contributions_1[k,i]+attribution2$Absolute_Contributions_1[k,i])/4 + Hx_1[i+1]) +
              lx_1[i] *( (1-sum(attribution2$Absolute_Contributions_1[(k+1):nrow(attribution2$Absolute_Contributions_1),i]))/2
                         - (attribution1$Absolute_Contributions_1[k,i]+attribution2$Absolute_Contributions_1[k,i])/4 + Hx_2[i+1])
          )
      }
      else if(k==nrow(attribution1$Absolute_Contributions_1)){
        mortality_Andreev[k,i] = 0.5 * (attribution1$Absolute_Contributions_2[k,i]-attribution2$Absolute_Contributions_2[k,i])*
          (
            lx_2[i] *( (1-sum(attribution2$Absolute_Contributions_1[2:(k-1),i]))/2 -
                         (attribution1$Absolute_Contributions_1[k,i]+attribution2$Absolute_Contributions_1[k,i])/4 + Hx_1[i+1]) +
              lx_1[i] *( (1-sum(attribution1$Absolute_Contributions_1[2:(k-1),i]))/2 -
                           (attribution1$Absolute_Contributions_1[k,i]+attribution2$Absolute_Contributions_1[k,i])/4 + Hx_2[i+1])
          )
      }
      else{
        mortality_Andreev[k,i] = 0.5 * (attribution1$Absolute_Contributions_2[k,i]-attribution2$Absolute_Contributions_2[k,i])*
          (
            lx_2[i] *( (1-sum(attribution2$Absolute_Contributions_1[2:(k-1),i])-sum(attribution1$Absolute_Contributions_1[(k+1):nrow(attribution1$Absolute_Contributions_1),i]))/2 -
                         (attribution1$Absolute_Contributions_1[k,i]+attribution2$Absolute_Contributions_1[k,i])/4 + Hx_1[i+1]) +
              lx_1[i] *( (1-sum(attribution1$Absolute_Contributions_1[2:(k-1),i])-sum(attribution2$Absolute_Contributions_1[(k+1):nrow(attribution2$Absolute_Contributions_2),i]))/2 -
                           (attribution1$Absolute_Contributions_1[k,i]+attribution2$Absolute_Contributions_1[k,i])/4 + Hx_2[i+1])
          )
      }}
    mortality_Andreev[1,i]=sum(mortality_Andreev[2:nrow(attribution1$Absolute_Contributions_1),i])
  }


  for (i in c(1:t)){
    for(k in c(2:nrow(attribution1$Absolute_Contributions_1))){
      if(k==2){
        disability_Andreev[k,i] = 0.5 * (attribution1$Absolute_Contributions_1[k,i]-attribution2$Absolute_Contributions_1[k,i])*
          (
            lx_2[i] * (1-(sum(attribution1$Absolute_Contributions_2[(k+1):nrow(attribution1$Absolute_Contributions_2),i]))/2 -
                         (attribution1$Absolute_Contributions_2[k,i]+attribution2$Absolute_Contributions_2[k,i])/4) +
              lx_1[i] * (1-(sum(attribution2$Absolute_Contributions_2[(k+1):nrow(attribution1$Absolute_Contributions_2),i]))/2 -
                           (attribution1$Absolute_Contributions_2[k,i]+attribution2$Absolute_Contributions_2[k,i])/4)
          )
      }
      else if(k==nrow(attribution1$Absolute_Contributions_2)){
        disability_Andreev[k,i] = 0.5 * (attribution1$Absolute_Contributions_1[k,i]-attribution2$Absolute_Contributions_1[k,i])*
          (
            lx_2[i] * (1-(sum(attribution2$Absolute_Contributions_2[2:(k-1),i]))/2 -
                         (attribution1$Absolute_Contributions_2[k,i]+attribution2$Absolute_Contributions_2[k,i])/4) +
              lx_1[i] * (1-(sum(attribution1$Absolute_Contributions_2[2:(k-1),i]))/2 -
                           (attribution1$Absolute_Contributions_2[k,i]+attribution2$Absolute_Contributions_2[k,i])/4)
          )
      }else{
        disability_Andreev[k,i] = 0.5 * (attribution1$Absolute_Contributions_1[k,i]-attribution2$Absolute_Contributions_1[k,i])*
          (
            lx_2[i] * (1-(sum(attribution2$Absolute_Contributions_2[2:(k-1),i])+sum(attribution1$Absolute_Contributions_2[(k+1):nrow(attribution1$Absolute_Contributions_2),i]))/2 -
                         (attribution1$Absolute_Contributions_2[k,i]+attribution2$Absolute_Contributions_2[k,i])/4) +
              lx_1[i] * (1-(sum(attribution1$Absolute_Contributions_2[2:(k-1),i])+sum(attribution2$Absolute_Contributions_2[(k+1):nrow(attribution1$Absolute_Contributions_2),i]))/2 -
                           (attribution1$Absolute_Contributions_2[k,i]+attribution2$Absolute_Contributions_2[k,i])/4)
          )
      }
    }
    disability_Andreev[1,i]=sum(disability_Andreev[2:nrow(attribution1$Absolute_Contributions_1),i])
  }
  total_Andreev <- disability_Andreev + mortality_Andreev

  colnames(mortality_Andreev) <- colnames(disability_Andreev)  <- colnames(total_Andreev) <- c(1:t)
  rownames(mortality_Andreev) <- rownames(disability_Andreev)  <- rownames(total_Andreev) <-c("total","backgroud",attribution1$var_list)


  Andreev <- list(mortality_effect = mortality_Andreev, disability_effect = disability_Andreev, total_effect=total_Andreev,
                  population1_HE=Hx_1[1],population2_HE=Hx_2[1],population1_LE=ex_1[1],population2_LE=ex_2[1])

  Andreev$var_list <- attribution1$var_list
  Andreev$time_list <- 1:t
  Andreev$copula <- "Decomposition_sullivan"
  class(Andreev) <- "LongDecompHE"
  return(Andreev)
}

