% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MVOPR.R
\name{MVOPR3}
\alias{MVOPR3}
\title{Multi-View Orthogonal Projection Regression for three modalities}
\usage{
MVOPR3(
  M1,
  M2,
  M3,
  Y,
  RRR_Control = list(Sparsity = TRUE, nrank = 10, ic.type = "GIC"),
  family = "gaussian",
  penalty = "lasso"
)
}
\arguments{
\item{M1}{A numeric matrix (n x p1) for the first modality.}

\item{M2}{A numeric matrix (n x p2) for the second modality. Assumes `M2` is correlated to `M1` via a low-rank matrix.}

\item{M3}{A numeric matrix (n x p3) for the third modality. Assumes `M3` is correlated to `M1` and `M2` via a low-rank matrix.}

\item{Y}{A numeric response vector of length `n`, connected to `M1`, `M2`, and `M3`.}

\item{RRR_Control}{A list to control the fitting for reduced rank regression.
\describe{
    \item{\code{Sparsity}}{Logical. If `TRUE`, performs Sparse Orthogonal Factor Regression (SOFAR); otherwise, a reduced-rank regression model is fitted.}
    \item{\code{nrank}}{Integer. Maximum rank to be searched for the reduced-rank model.}
    \item{\code{ic.type}}{Character. Model selection criterion: `"AIC"`, `"BIC"`, or `"GIC"`.}
  }}

\item{family}{Either "gaussian", "binomial", or "poisson", depending on the response.}

\item{penalty}{The penalty to be applied in the outcome model Y to M1 and M2. Either "MCP" (the default), "SCAD", or "lasso".}
}
\value{
A list containing:
\describe{
  \item{\code{fitY}}{A fitted object from `cv.ncvreg`, containing the penalized regression results for `Y`.}
  \item{\code{fitM2}}{The fitted reduced-rank regression (`sofar` or `rrr` object) for `M2` given `M1`.}
  \item{\code{fitM3}}{The fitted reduced-rank regression (`sofar` or `rrr` object) for `M3` given `M1` and `M2`.}
  \item{\code{CoefY}}{A vector of estimated regression coefficients for `Y`.}
  \item{\code{coefM2}}{A matrix of estimated regression coefficients for `M2` given `M1`.}
  \item{\code{coefM3}}{A matrix of estimated regression coefficients for `M3` given `M1` and `M2`.}
  \item{\code{rank1}}{An integer indicating the estimated rank of the reduced-rank regression for `M2`.}
  \item{\code{rank2}}{An integer indicating the estimated rank of the reduced-rank regression for `M3`.}
  \item{\code{P1}}{A projection matrix used to extract the orthogonal components of `M1`.}
  \item{\code{P2}}{A projection matrix used to extract the orthogonal components of `E2`, which is the error term in the regression for `M2` given `M1`.}
  \item{\code{M1s}}{A transformed version of `M1` after projection.}
  \item{\code{M2s}}{A transformed version of `M2` after removing the effect of `M1` and projecting to the orthogonal space.}
  \item{\code{M3s}}{A transformed version of `M3` after removing the effects of `M1` and `M2`.}
}

#' @references
Dai, Z., Huang, Y. J., & Li, G. (2025). Multi-View Orthogonal Projection Regression with Application in Multi-omics Integration. arXiv preprint arXiv:2503.16807. Available at <https://arxiv.org/abs/2503.16807>
}
\description{
Fit Multi-View Orthogonal Projection Regression for three modalities with Lasso, MCP, SCAD. The function is capable for linear, logistic, and poisson regression.
}
\examples{

## Simulation: three modalities
p1 = 50; p2 = 50; p3 = 50; n = 200
rank = 2

beta = c(rep(c(rep(1,5),rep(0,45)),3))
M1 = matrix(rnorm(p1*n),n,p1)

U1 = matrix(rnorm(rank*p1),p1,rank)
V1 = matrix(runif(rank*p2,-0.1,0.1),rank,p2)
B1 = U1 \%*\% V1

U2 = matrix(rnorm(rank*p1),p1,rank)
V2 = matrix(runif(rank*p2,-0.1,0.1),rank,p3)
B2 = U2 \%*\% V2

U3 = matrix(rnorm(rank*p2),p2,rank)
V3 = matrix(runif(rank*p2,-0.1,0.1),rank,p3)
B3 = U3 \%*\% V3

E1 = matrix(rnorm(p2*n),n,p2)
E2 = matrix(rnorm(p3*n),n,p3)

M2 = M1 \%*\% B1 + E1
M3 = M1 \%*\% B2 + M2 \%*\% B3 + E2
Y = cbind(M1,M2,M3) \%*\% matrix(beta,p1+p2+p3,1)

## Fit MVOPR with Lasso
Fit1 = MVOPR3(M1,M2,M3,Y,RRR_Control = list(Sparsity = FALSE),penalty = 'lasso')

## Fit MVOPR with MCP
Fit2 = MVOPR3(M1,M2,M3,Y,RRR_Control = list(Sparsity = FALSE),penalty = 'MCP')

## Fit MVOPR with SCAD
Fit3 = MVOPR3(M1,M2,M3,Y,RRR_Control = list(Sparsity = FALSE),penalty = 'SCAD')

## Compare the variable selection between Lasso, MCP, SCAD
print(data.frame(Lasso = Fit1$CoefY[2:151],MCP = Fit2$CoefY[2:151],SCAD = Fit3$CoefY[2:151],beta))

}
