# =============================================================================
# FULL ECF ESTIMATION PIPELINE
# =============================================================================


#' Estimate alpha and gamma from ECF modulus
#'
#' @param t_grid Vector of frequency values.
#' @param phi_vals Complex vector of ECF values.
#' @param weights Optional vector of weights.
#' @return A list with elements \code{alpha_hat} and \code{gamma_hat}.
#' @importFrom stats lm
#' @export
estimate_alpha_gamma <- function(t_grid, phi_vals, weights = NULL) {
  y <- log(-log(Mod(phi_vals)))
  X <- log(abs(t_grid))

  if (!is.null(weights)) {
    reg <- lm(y ~ X, weights = weights)
  } else {
    reg <- lm(y ~ X)
  }

  alpha_hat <- coef(reg)[2]
  a_hat <- coef(reg)[1]
  gamma_hat <- exp(a_hat / alpha_hat)

  return(list(alpha_hat = alpha_hat, gamma_hat = gamma_hat))
}


#' Estimate beta and delta from ECF phase
#'
#' @param t_grid Vector of frequency values.
#' @param phi_vals Complex vector of ECF values.
#' @param alpha_hat Estimated alpha.
#' @param gamma_hat Estimated gamma.
#' @param weights Optional vector of weights.
#' @return A list with elements \code{beta_hat} and \code{delta_hat}.
#' @importFrom stats lm
#' @export
estimate_beta_delta <- function(t_grid, phi_vals, alpha_hat, gamma_hat, weights = NULL) {
  z_k <- Arg(phi_vals)
  eta_k <- eta0(t_grid, alpha_hat, gamma_hat)
  B_k <- gamma_hat^alpha_hat * eta_k
  A <- cbind(B_k, t_grid)

  if (!is.null(weights)) {
    reg <- lm(z_k ~ A - 1, weights = weights)
  } else {
    reg <- lm(z_k ~ A - 1)
  }

  coeffs <- coef(reg)
  beta_hat <- coeffs[1]
  delta_hat <- coeffs[2]

  return(list(beta_hat = beta_hat, delta_hat = delta_hat))
}


#' Estimate all stable parameters from empirical characteristic function
#'
#' @param X Numeric vector of data.
#' @param t_grid Optional vector of frequencies (default: seq(0.1, 1.0, length.out = 50)).
#' @param weights Optional vector of weights.
#' @return A list containing estimated parameters: \code{alpha}, \code{beta},
#'   \code{gamma}, and \code{delta}.
#' @export
ecf_estimate_all <- function(X, t_grid = NULL, weights = NULL) {
  if (is.null(t_grid)) {
    t_grid <- seq(0.1, 1.0, length.out = 50)
  }

  phi_vals <- ecf_empirical(X, t_grid)

  ag_result <- estimate_alpha_gamma(t_grid, phi_vals, weights)
  alpha_hat <- ag_result$alpha_hat
  gamma_hat <- ag_result$gamma_hat

  bd_result <- estimate_beta_delta(t_grid, phi_vals, alpha_hat, gamma_hat, weights)
  beta_hat <- bd_result$beta_hat
  delta_hat <- bd_result$delta_hat

  return(list(
    alpha = alpha_hat,
    beta = beta_hat,
    gamma = gamma_hat,
    delta = delta_hat
  ))
}
