# =============================================================================
# QUANTILE-BASED RATIOS
# =============================================================================


#' Compute McCulloch quantile ratios from sample data
#'
#' This function calculates four key ratios used in McCulloch's method for
#' estimating stable distribution parameters:
#' - v_alpha: shape indicator
#' - v_beta: skewness indicator
#' - v_gamma: scale proxy
#' - v_delta: location proxy
#'
#' @param X Numeric vector of sample data.
#' @return A list containing v_alpha, v_beta, v_gamma, and v_delta.
#' @export
compute_quantile_ratios <- function(X) {
  x05 <- quantile(X, 0.05, na.rm = TRUE)
  x25 <- quantile(X, 0.25, na.rm = TRUE)
  x50 <- quantile(X, 0.5, na.rm = TRUE)
  x75 <- quantile(X, 0.75, na.rm = TRUE)
  x95 <- quantile(X, 0.95, na.rm = TRUE)

  v_alpha <- (x95 - x05) / (x75 - x25)
  v_beta  <- (x05 + x95 - 2 * x50) / (x95 - x05)
  v_gamma <- x75 - x25
  v_delta <- -x50

  return(list(
    v_alpha = v_alpha,
    v_beta  = v_beta,
    v_gamma = v_gamma,
    v_delta = v_delta
  ))
}
