% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simregions.R
\name{simregions}
\alias{simregions}
\alias{plot.regions_sim}
\title{Simulate regions data}
\usage{
simregions(
  nvert,
  nregions,
  nvar = 1,
  r2 = 0.95,
  minvert = 3,
  cont = TRUE,
  sl.dif = 0
)

\method{plot}{regions_sim}(x, scores = 1, lines = TRUE, ...)
}
\arguments{
\item{nvert}{\code{numeric}; the number of vertebrae for which to simulate data.}

\item{nregions}{\code{numeric}; the desired number of regions in the simulated data.}

\item{nvar}{\code{numeric}; the number of PCO axes to simulate. Default is 1.}

\item{r2}{\code{numeric}; a vector containing the \eqn{R^2} of the true segmented regression model for each simulated PCO. If a single value is supplied, all PCOs will receive that value. Otherwise, one value should be supplied for each simulated PCO.}

\item{minvert}{\code{numeric}; the minimum number of vertebrae allowed in each region. Default is 3.}

\item{cont}{\code{logical}; whether to use models that are continuous (\code{TRUE}) or discontinuous (\code{FALSE}) at the breakpoints to generate the data. Default is \code{TRUE}.}

\item{sl.dif}{\code{numeric}; the minimum required difference in slopes between adjacent regions, expressed as a proportion of the maximal difference between allowable slopes. Must be between 0 and 1. See Details.}

\item{x}{a \code{regions_sim} object.}

\item{scores}{\code{numeric}; for which simulated PCO scores the simulated values should be plotted.}

\item{lines}{\code{logical}; whether to display the simulated regression lines on the plot. Default is \code{TRUE}.}

\item{\dots}{ignored.}
}
\value{
\code{simregions()} returns a \code{regions_sim} object, which contains the vertebra indices in the \code{Xvar} entry, the PCO scores in the \code{Yvar} entry, the simulated breakpoints in the \code{BPs} entry, the simulated model coefficients in the \code{coefs} entry, and the simulated error standard deviation in the \code{ersd} entry. The attribute \code{"design"} contains the design matrix, which when multiplied by the coefficients and added to a random normal variate with standard deviation equal to the error standard deviation yields the observed PCO scores.

\code{plot()} returns a \code{ggplot} object that can be manipulated using \code{ggplot2} syntax. The plot is similar to that produced by \code{\link[=plot.regions_pco]{plot.regions_pco()}} and to that produced by \code{\link[=plotsegreg]{plotsegreg()}} except that the displayed lines (if requested) are the true rather than fitted regression lines.
}
\description{
\code{simregions()} simulates vertebrae and PCOs that satisfy certain constraints.
}
\details{
\code{simregions()} generates PCO scores for each requested vertebra such that certain conditions are met. The slopes for each region are drawn from a uniform distribution with limits of -.5 and .5. If a set of slopes contains two adjacent slopes that have a difference less than \code{sl.dif}, it is rejected and a new one is drawn. The scaling of the PCOs is determined by the slopes and the requested \eqn{R^2}. The PCOs will not necessarily be in order from most variable to least variable as they are in a traditional PCO analysis.

Intercepts (the intercept of the first region when \code{cont = TRUE} and the intercept of all regions when \code{cont = FALSE}) are drawn from a uniform distribution with limits of \eqn{-n/4} and \eqn{n/4}, where \eqn{n} is the number of breakpoints, one less than \code{nregions}. Intercepts other than the first when \code{cont = TRUE} are determined by the slopes.

The \code{cont}, \code{r2}, and \code{sl.dif} arguments control how easy it is for fitted segmented regression models to capture the true structure. When \code{cont = TRUE}, it can be harder to determine exactly where regions begin and end, especially if \code{sl.dif} is 0. When \code{r2} is high, there is little variation around the true line, so the fitted lines will be more precise and region boundaries clearer. When \code{sl.dif} is high, slopes of adjacent regions are different from each other, so it is easier to detect region boundaries. Setting \code{sl.dif} to between .5 and 1 ensures that the slopes in adjacent regions have different signs.
}
\examples{
# Simulate 40 vertebra, 4 regions (3 breakpoints), 3 PCOs,
# true model R2 of .9, continuous
set.seed(11)
sim <- simregions(nvert = 30, nregions = 4, nvar = 3, r2 = .95,
                  minvert = 3, cont = TRUE)

sim

# Plot the true data-generating lines and breakpoints
plot(sim, scores = 1:3)

# Run segmented regression models on simulated data,
# up to 6 regions
simresults <- calcregions(sim, scores = 1:3, noregions = 6,
                          minvert = 3, cont = TRUE,
                          verbose = FALSE)

summary(simresults)

# Select best model for each number of regions
(simmodels <- modelselect(simresults))

# Evaluate support for each model and rank models
(simsupp <- modelsupport(simmodels))
# AICc supports 3-4 regions

# Evaluate model performance of best model
modelperf(sim, modelsupport = simsupp,
          criterion = "aic", model = 1)
# Second best model (3 regions) does quite well, too
modelperf(sim, modelsupport = simsupp,
          criterion = "aic", model = 2)

#Plot best model fit
plotsegreg(sim, scores = 1:3,
           modelsupport = simsupp,
           criterion = "aic", model = 1)

# Calculate variability of estimate breakpoints for
# 3-region model; high uncertainty for breakpoints
# 1 and 2. Note weighted value for breakpoint 2
# differs from that of best model
bpvar <- calcBPvar(simresults, noregions = 4,
                   pct = .05, criterion = "aic")
bpvar

# Plot estimated vertebral map with variability
plotvertmap(sim, modelsupport = simsupp, model = 1,
            criterion = "aic", text = TRUE)

# True map; pretty close
plotvertmap(sim, bps = c(3, 7, 24),
            text = TRUE)
}
\seealso{
\code{\link[=calcregions]{calcregions()}} for fitting segmented regression models to the simulated data; \code{\link[=calcmodel]{calcmodel()}} for fitting a single segmented regression model to the simulated data; \code{\link[=plotsegreg]{plotsegreg()}} for plotting estimated regression lines.
}
