% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/networkProperties.R
\name{networkProperties}
\alias{networkProperties}
\title{Calculate the topological properties for a network module}
\usage{
networkProperties(
  network,
  data,
  correlation,
  moduleAssignments = NULL,
  modules = NULL,
  backgroundLabel = "0",
  discovery = NULL,
  test = NULL,
  simplify = TRUE,
  verbose = TRUE
)
}
\arguments{
\item{network}{a list of interaction networks, one for each dataset. Each 
entry of the list should be a \eqn{n * n} matrix or where each element 
contains the edge weight between nodes \eqn{i} and \eqn{j} in the inferred 
network for that dataset.}

\item{data}{a list of matrices, one for each dataset. Each entry of the list 
should be the data used to infer the interaction \code{network} for that 
dataset. The columns should correspond to variables in the data
(nodes in the network) and rows to samples in that dataset.}

\item{correlation}{a list of matrices, one for each dataset. Each entry of
the list should be a \eqn{n * n} matrix where each element contains the 
correlation coefficient between nodes \eqn{i} and \eqn{j} in the 
\code{data} used to infer the interaction network for that dataset.}

\item{moduleAssignments}{a list of vectors, one for each \emph{discovery} 
dataset, containing the module assignments for each node in that dataset.}

\item{modules}{a list of vectors, one for each \code{discovery} dataset, 
of modules to perform the analysis on. If unspecified, all modules
in each \code{discovery} dataset will be analysed, with the exception of 
those specified in \code{backgroundLabel} argument.}

\item{backgroundLabel}{a single label given to nodes that do not belong to 
any module in the \code{moduleAssignments} argument. Defaults to "0". Set 
to \code{NULL} if you do not want to skip the network background module.}

\item{discovery}{a vector of names or indices denoting the \emph{discovery}
dataset(s) in the \code{data}, \code{correlation}, \code{network}, 
\code{moduleAssignments}, \code{modules}, and \code{test} lists.}

\item{test}{a list of vectors, one for each \code{discovery} dataset,
of names or indices denoting the \emph{test} dataset(s) in the \code{data}, 
\code{correlation}, and \code{network} lists.}

\item{simplify}{logical; if \code{TRUE}, simplify the structure of the output
list if possible (see Return Value).}

\item{verbose}{logical; should progress be reported? Default is \code{TRUE}.}
}
\value{
A nested list structure. At the top level, the list has one element per 
 \code{'discovery'} dataset. Each of these elements is a list that has one
 element per \code{'test'} dataset analysed for that \code{'discovery'} 
 dataset. Each of these elements is a list that has one element per 
 \code{'modules'} specified. Each of these is a list containing the following  
 objects:
 \describe{
   \item{\code{'degree'}:}{
     The weighted within-module degree: the sum of edge weights for each 
     node in the module.
   }
   \item{\code{'avgWeight'}:}{
     The average edge weight within the module.
   }
 }
 If the \code{'data'} used to infer the \code{'test'} network is provided  
 then the following are also returned:
 \describe{
   \item{\code{'summary'}:}{
     A vector summarising the module across each sample. This is calculated 
     as the first eigenvector of the module from a principal component 
     analysis.
   }
   \item{\code{'contribution'}:}{
     The \emph{node contribution}: the similarity between each node and the
     \emph{module summary profile} (\code{'summary'}).
   }
   \item{\code{'coherence'}:}{
     The proportion of module variance explained by the \code{'summary'}
     vector.
   }
 }
 When \code{simplify = TRUE} then the simplest possible structure will be 
 returned. E.g. if the network properties are requested for only one module 
 in only one dataset, then the returned list will have only the above elements. 
 
 When \code{simplify = FALSE} then a nested list of datasets will always be 
 returned, i.e. each element at the top level and second level correspond to
 a dataset, and each element at the third level will correspond to modules 
 discovered in the dataset specified at the top level if module labels are 
 provided in the corresponding \code{moduleAssignments} list element. E.g. 
 \code{results[["Dataset1"]][["Dataset2"]][["module1"]]} will contain the 
 properties of "module1" as calculated in "Dataset2", where "module1" was 
 indentified in "Dataset1". Modules and datasets for which calculation of 
 the network properties have not been requested will contain \code{NULL}.
}
\description{
Calculates the network properties used to assess module preservation for one
or more modules in a user specified dataset.
}
\details{
\subsection{Input data structures:}{
  The \link[=modulePreservation]{preservation of network modules} in a second
  dataset is quantified by measuring the preservation of topological
  properties between the \emph{discovery} and \emph{test} datasets. These 
  properties are calculated not only from the interaction networks inferred
  in each dataset, but also from the data used to infer those networks (e.g.
  gene expression data) as well as the correlation structure between 
  variables/nodes. Thus, all functions in the \code{NetRep} package have the 
  following arguments: 
  \describe{
    \item{\code{network}:}{
      a list of interaction networks, one for each dataset.
    }
    \item{\code{data}:}{
      a list of data matrices used to infer those networks, one for each 
      dataset.
    }
    \item{\code{correlation}:}{
     a list of matrices containing the pairwise correlation coefficients 
     between variables/nodes in each dataset.
    } 
    \item{\code{moduleAssignments}:}{
     a list of vectors, one for each \emph{discovery} dataset, containing 
     the module assignments for each node in that dataset.
    }
    \item{\code{modules}:}{
     a list of vectors, one for each \emph{discovery} dataset, containing
     the names of the modules from that dataset to analyse.  
    }
    \item{\code{discovery}:}{
      a vector indicating the names or indices of the previous arguments' 
      lists to use as the \emph{discovery} dataset(s) for the analyses.
    }
    \item{\code{test}:}{
      a list of vectors, one vector for each \emph{discovery} dataset, 
      containing the names or indices of the \code{network}, \code{data}, and 
      \code{correlation} argument lists to use as the \emph{test} dataset(s) 
      for the analysis of each \emph{discovery} dataset.
    }
  }
  
  The formatting of these arguments is not strict: each function will attempt
  to make sense of the user input. For example, if there is only one 
  \code{discovery} dataset, then input to the \code{moduleAssigments} and 
  \code{test} arguments may be vectors, rather than lists. If the 
  \code{networkProperties} are being calculate within the \emph{discovery} or
  \emph{test} datasets, then the \code{discovery} and \code{test} arguments do
  not need to be specified, and the input matrices for the \code{network},
  \code{data}, and \code{correlation} arguments do not need to be wrapped in
  a list.
}
\subsection{Analysing large datasets:}{
  Matrices in the \code{network}, \code{data}, and \code{correlation} lists
  can be supplied as \code{\link{disk.matrix}} objects. This class allows 
  matrix data to be kept on disk and loaded as required by \pkg{NetRep}. 
  This dramatically decreases memory usage: the matrices for only one 
  dataset will be kept in RAM at any point in time.
}
}
\examples{
# load in example data, correlation, and network matrices for a discovery and test dataset:
data("NetRep")

# Set up input lists for each input matrix type across datasets. The list
# elements can have any names, so long as they are consistent between the
# inputs.
network_list <- list(discovery=discovery_network, test=test_network)
data_list <- list(discovery=discovery_data, test=test_data)
correlation_list <- list(discovery=discovery_correlation, test=test_correlation)
labels_list <- list(discovery=module_labels)

# Calculate the topological properties of all network modules in the discovery dataset
props <- networkProperties(
  network=network_list, data=data_list, correlation=correlation_list, 
  moduleAssignments=labels_list
)
  
# Calculate the topological properties in the test dataset for the same modules
test_props <- networkProperties(
  network=network_list, data=data_list, correlation=correlation_list,
  moduleAssignments=labels_list, discovery="discovery", test="test"
)

}
\seealso{
\link[=nodeOrder]{Getting nodes ordered by degree.}, and
  \link[=sampleOrder]{Ordering samples by module summary}
}
