#
#  growth : A Library of Normal Distribution Growth Curve Models
#  Copyright (C) 1998 J.K. Lindsey
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
#  SYNOPSIS
#
#     potthoff(y, x, cov=NULL, times=NULL, torder=0, orthog=T)
#
#  DESCRIPTION
#
#    Function to fit the Potthoff and Roy (1964) growth curve model
# Y contains responses, X between unit design matrix, and Z within
# unit matrix

potthoff <- function(y, x, cov=NULL, times=NULL, torder=0, orthog=T){
pcov <- function(z, s, x){
	p <- t(x)%**%z
	solve(p%*%(s%**%diag(n))%*%t(p))}
call <- sys.call()
if(is.data.frame(y))y <- as.matrix(y)
n <- nrow(y)
r <- ncol(y)
if(is.null(times))times <- 1:r
if(orthog)z <- orth(times,order=r-1)
else {
	z <- rep(1,r)
	tt <- times-sum(times)/r
	z <- rbind(z,tt)
	for(i in 2:(r-1))z <- rbind(z,tt^i)}
if(is.null(cov)) xx <- matrix(rep(1,n),ncol=1)
else xx <- x
s <- t(y)%*%(diag(n)-xx%*%solve(t(xx)%*%xx)%*%t(xx))%*%y/n
ss <- solve(t(y)%*%(diag(n)-x%*%solve(t(x)%*%x)%*%t(x))%*%y/n)
b <- solve(t(xx)%*%xx)%*%t(xx)%*%y
if(!is.matrix(b))b <- matrix(b,nrow=1)
if(!is.numeric(torder)){
	zz <- diag(r)
	b1 <- b
	s1 <- s}
else {
	zz <- z[1:(torder+1),]
	if(!is.matrix(zz))zz <- matrix(zz,nrow=1)
	b1 <- b%*%ss%*%t(zz)%*%solve(zz%*%ss%*%t(zz))
	s1 <- y-xx%*%b1%*%zz
	s1 <- t(s1)%*%s1/n}
res <- y-xx%*%b1%*%zz
if(!is.matrix(b1))b1 <- matrix(b1,nrow=1)
like <- n*(r*(1+log(2*pi))+log(det(s1)))/2
aic <- like+length(b1)+r*(r+1)/2
pc <- pcov(zz,solve(s1),xx)
if(is.matrix(pc)){
	d <- sqrt(diag(pc))
	se <- matrix(d,ncol=ncol(b1),byrow=T)}
else se <- sqrt(pc)
corr <- pc
if(is.matrix(pc))
	for(i in 1:ncol(pc))for(j in 1:nrow(pc))corr[i,j] <- pc[i,j]/d[i]/d[j]
if(!is.numeric(torder)){
	if(is.null(colnames(y)))colnames(b1) <- paste("t",1:ncol(b1),sep="")
	else colnames(b1) <- colnames(y)
	if(is.null(colnames(y)))colnames(se) <- paste("t",1:ncol(se),sep="")
	else colnames(se) <- colnames(y)}
if(is.matrix(b1)&dim(b1)[1]>1){
	if(is.null(colnames(cov))){
		tn <- "Int"
		if(ncol(cov)>1)tn <- c(tn,paste("cov",1:(ncol(cov)-1),sep=""))
		colnames(cov) <- tn}
	rownames(b1) <- colnames(cov)}
else {
	b1 <- matrix(b1,nrow=1)
	rownames(b1) <- "Mean"}
if(is.matrix(se)&&dim(se)[1]>1)rownames(se) <- colnames(cov)
else {
	se <- matrix(se,nrow=1)
	rownames(se) <- "Mean"}
d <- sqrt(diag(s1))
c1 <- s1
for(i in 2:r)for(j in 1:(i-1))c1[i,j] <- s1[i,j]/d[i]/d[j]
z <- list(
	call=call,
	y=y,
	x=x,
	time=z,
	torder=torder,
	ns=n,
	nt=n*r,
	df=n*r-(length(b1)+r*(r+1)/2),
	beta=b1,
	cov=s1,
	maxlike=like,
	aic=aic,
	pcov=pc,
	pcorr=corr,
	se=se,
	corr=c1,
	residuals=res)
class(z) <- "potthoff"
return(z)}

coefficients.potthoff <- function(z) z$beta
deviance.potthoff <- function(z) 2*z$maxlike
residuals.potthoff <- function(z) z$residuals

print.potthoff <- function(z, digits = max(3, .Options$digits - 3)) {
	cat("\nCall:\n",deparse(z$call),"\n\n",sep="")
	cat("Number of subjects    ",z$ns,"\n")
	cat("Number of observations",z$nt,"\n")
	cat("-Log likelihood   ",z$maxlike,"\n")
	cat("Degrees of freedom",z$df,"\n")
	cat("AIC               ",z$aic,"\n\n")
	if(is.null(colnames(z$beta))&z$torder<=4){
		tn <- "Int"
		if(z$torder>0)tn <- c(tn,paste("t^",1:z$torder,sep=""))
		colnames(z$beta) <- tn
		colnames(z$se) <- tn}
	cat("Estimates of linear parameters\n")
	print(z$beta)
	cat("Standard errors\n")
	print(z$se)
	nlp <- length(z$beta)
	if(nlp>1){
		cat("\nCorrelation matrix of linear parameters\n")
		dimnames(z$pcorr) <- list(seq(1,nlp),seq(1,nlp))
		print.default(z$pcorr, digits=digits)}
	cat("\nCovariance/correlation matrix\n")
	if(is.null(colnames(z$corr))){
		tn <- paste("t",1:ncol(z$corr),sep="")
		dimnames(z$corr) <- list(tn,tn)}
	print(z$corr)}
