% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CRR.R
\name{CRR}
\alias{CRR}
\title{Estimate cumulative response rates (CRR) and test their equality between two groups}
\usage{
CRR(
  t2PROGRESSION,
  STATUS_PROGRESSION,
  t2RESPONSE,
  STATUS_RESPONSE,
  TRT,
  time = NULL,
  alpha = 0.95
)
}
\arguments{
\item{t2PROGRESSION}{time to progression/death or censoring}

\item{STATUS_PROGRESSION}{binary indicator for progression status: 1 for progression/death; 0 for censoring}

\item{t2RESPONSE}{time to response or censoring}

\item{STATUS_RESPONSE}{binary indicator for response status: 1 for response; 0 for censoring}

\item{TRT}{binary indicator for treatment assignment: 1 for treatment arm and 0 for control arm}

\item{time}{user-selected time points at which the cumulative response rate is to be estimated; the default value is "NULL" and the cumulative response rate will be estimated at all observed time points}

\item{alpha}{coverage level of the point-wise confidence interval for the cumulative response rate; the default value is 0.95}
}
\value{
A list with following elements
\itemize{
\item{result0:}   a data matrix containing "time", "CRR estimates (group 0)", "standard error of CRR estimates (group 0)",  "confidence interval of CRR (group 0)"
\item{result1:}   a data matrix containing "time", "CRR estimates (group 1)", "standard error of CRR estimates (group 1)",  "confidence interval of CRR (group 1)"
\item{pvalue:}    the p-value from two group comparison
}
}
\description{
Estimate cumulative response rates (CRR) and test their equality between two groups
}
\examples{

library(cmprsk)
n=100
set.seed(10)

# Generate the data

trt=rbinom(n, 1, 0.5)
error=rnorm(n)
tr=exp(rnorm(n)+error-trt*0.5+0.5)
tp=exp(rnorm(n)+error+trt*0.25)
tr[tp<tr]=Inf
tc=runif(n, 3, 8.5)
t2response=pmin(tr, tc)
delta_response=1*(tr<tc)
t2progression=pmin(tp, tc)
delta_progression=1*(tp<tc)

# Estimate the PBIR in two groups

fit=CRR(t2PROGRESSION=t2progression,
         STATUS_PROGRESSION=delta_progression,
         t2RESPONSE=t2response,
         STATUS_RESPONSE=delta_response,
         TRT=trt)

fit

# Plot the estimated PBIR by group

tt1=c(0, fit$result1$time)
CRR1=c(0, fit$result1$CRR)
B1=length(tt1)
tt1=rep(tt1, rep(2, B1))[-1]
CRR1=rep(CRR1, rep(2, B1))[-(2*B1)]
tt0=c(0, fit$result0$time)
CRR0=c(0, fit$result0$CRR)
B0=length(tt0)
tt0=rep(tt0, rep(2, B0))[-1]
CRR0=rep(CRR0, rep(2, B0))[-(2*B0)]
plot(range(c(fit$result1$time, fit$result0$time)),
     range(c(fit$result1$CRR, fit$result0$CRR)),
     xlab="time",  ylab="CRR",
     main="black: group 0; red: group 1", type="n")
lines(tt0, CRR0, col=1)
lines(tt1, CRR1, col=2)
}
\references{
Gray, RJ. (1988) A class of K-sample tests for comparing the cumulative incidence of a competing risk, ANNALS OF STATISTICS, 16:1141-1154.

Aalen, O. (1978) Nonparametric estimation of partial transition probabilities in multiple decrement models, ANNALS OF STATISTICS, 6:534-545.
}
