#' Modified Sainte-Lague method for seats allocation.
#'
#' @param votes Un vector con los votos de cada partido.
#' @param seats Total de seats a repartir.
#' @param threshold Umbral electoral
#' @param div Divisor modificado en Modified Sainte-Lague
#' @param verbose Muestra los resultados en la consola
#' @return Un vector con los seats asignados a cada partido.
#' @examples
#' votes <- c(A = 100000, B = 75000, C = 50000, D = 25000)
#' saintelaguemod(votes, 10)
#' @export
saintelaguemod <- function(votes, seats, threshold = 0, div = 1.4, verbose = TRUE) {
  total_votes <- sum(votes)

  threshold_value <- total_votes * (threshold / 100)

  valid_votes <- votes[votes >= threshold_value]

  results <- numeric(length(valid_votes))
  names(results) <- names(valid_votes)

  divisors <- c(div, seq(3, by = 2, length.out = seats - 1))

  for (i in 1:seats) {
    quotients <- valid_votes / divisors[results + 1]  # Aplicar los divisores
    max_party <- which.max(quotients)
    results[max_party] <- results[max_party] + 1
  }

  allocated_seats <- results

  all_quotients <- list()
  for (party in names(valid_votes)) {
    quotients <- valid_votes[party] / divisors[1:seats]
    all_quotients[[party]] <- quotients
  }

  all_quotients_flat <- unlist(all_quotients)
  names(all_quotients_flat) <- rep(names(valid_votes), each = seats)

  sorted_quotients <- sort(all_quotients_flat, decreasing = TRUE)

  last_assigned_coc <- sorted_quotients[seats]
  first_non_assigned_coc <- sorted_quotients[seats + 1]

  party_to_surpass_index <- which(all_quotients_flat == first_non_assigned_coc)[1]
  party_to_surpass <- names(all_quotients_flat)[party_to_surpass_index]

  closest_party_index <- which(all_quotients_flat == last_assigned_coc)[1]
  closest_party <- names(all_quotients_flat)[closest_party_index]

  coc_difference <- last_assigned_coc - first_non_assigned_coc
  votes_needed <- ceiling((valid_votes[party_to_surpass] / first_non_assigned_coc) * coc_difference)

  allocated_seats_table <- data.frame(Partido = names(allocated_seats), Seats = allocated_seats)

  if (verbose) {
    message("Resultados de asignaci\u00f3n de esca\u00f1os:\n")
    print(allocated_seats_table)
    message("\n")
    message("El partido ", party_to_surpass, " necesitar\u00cda ", votes_needed,
            " votos m\u00e1s para quitarle un esca\u00f1o a ",
            closest_party, ".\n")
  }

  invisible(allocated_seats_table)
}
